<?php

//------------------------------------------

GFForms::include_addon_framework();

class GFTooltips extends GFAddOn {

	protected $_version = GFTT_VERSION;
	protected $_min_gravityforms_version = '2.0';

	protected $_slug = GFTT_SLUG;
	protected $_path = 'gf-tooltips/gf-tooltips.php';
	protected $_full_path = __FILE__;
	protected $_title = GFTT_NAME;
	protected $_short_title = 'Tooltips';
	protected $_url = 'https://jetsloth.com/gravity-forms-tooltips';

	/**
	 * Members plugin integration
	 */
	protected $_capabilities = array( 'gravityforms_edit_forms', 'gravityforms_edit_settings' );

	/**
	 * Permissions
	 */
	protected $_capabilities_settings_page = 'gravityforms_edit_settings';
	protected $_capabilities_form_settings = 'gravityforms_edit_forms';
	protected $_capabilities_uninstall = 'gravityforms_uninstall';

	private static $_instance = null;

	/**
	 * Get an instance of this class.
	 *
	 * @return GFTooltips
	 */
	public static function get_instance() {
		if ( self::$_instance == null ) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	private function __clone() {
	} /* do nothing */

	/**
	 * Handles anything which requires early initialization.
	 */
	public function pre_init() {
		parent::pre_init();
	}

	/**
	 * Handles hooks and loading of language files.
	 */
	public function init() {

		// inline css overrides. Run as late as possible
		add_action( 'gform_enqueue_scripts', array( $this, 'frontend_inline_styles' ), PHP_INT_MAX, 1 );

		add_filter( 'style_loader_tag', array( $this, 'configure_preload_assets' ), PHP_INT_MAX, 2 );

		add_filter( 'gform_field_content', array( $this, 'field_content' ), 100, 5 );

		parent::init();

	}

	/**
	 * Initialize the admin specific hooks.
	 */
	public function init_admin() {

		add_action( 'gform_editor_js', array( $this, 'editor_js' ) );

		if ( GFTT_GF_MIN_2_5 ) {
			add_filter( 'gform_field_settings_tabs', array( $this, 'custom_settings_tab' ), 10, 1 );
			add_action( 'gform_field_settings_tab_content_gftt', array( $this, 'custom_settings_markup' ), 10, 1 );
        }
		else {
			add_action( 'gform_field_standard_settings', array( $this, 'field_general_settings' ), 10, 2 );
        }

		add_filter( 'gform_tooltips', array( $this, 'field_tooltips' ) );

		add_action( 'admin_enqueue_scripts', array( $this, 'admin_inline_styles' ) );

		$name = plugin_basename($this->_path);
		add_action( 'after_plugin_row_'.$name, array( $this, 'gf_plugin_row' ), 10, 2 );

		parent::init_admin();

	}


	public function get_app_menu_icon() {
		return 'dashicons-admin-comments';
	}

	public function get_menu_icon() {
		return 'dashicons-admin-comments';
	}


	// # SCRIPTS & STYLES -----------------------------------------------------------------------------------------------


	/**
	 * Return the scripts which should be enqueued.
	 *
	 * @return array
	 */
	public function scripts() {
		$js_deps = array( 'jquery' );
		if ( wp_is_mobile() ) {
			$js_deps[] = 'jquery-touch-punch';
		}

		$scripts = array(
			array(
				'handle'   => 'gf_tooltips',
				'src'      => $this->get_base_url() . '/js/vendor/powertip.min.js',
				'version'  => $this->_version,
				'deps'     => $js_deps,
				'enqueue'  => array(
					array( 'admin_page' => array( 'form_editor', 'form_settings', 'plugin_settings' ) ),
					array( $this, 'maybe_enqueue_gf_tooltips_powertip_js' ),// front-end, any form with tooltips
				),
			),
			array(
				'handle'  => 'jetsloth_filters_actions',
				'src'     => $this->get_base_url() . '/js/jetsloth-filters-actions.js',
				'version' => $this->_version,
				'deps'     => array( 'jquery' ),
				'enqueue' => array(
					array( $this, 'maybe_enqueue_gf_tooltips_powertip_js' ),// front-end, any form with tooltips
				),
			),
			array(
				'handle'  => 'gf_tooltips_public',
				'src'     => $this->get_base_url() . '/js/gf_tooltip.js',
				'version' => $this->_version,
				//'deps'    => $js_deps,
                'strings' => array(
                    'version' => $this->_version
                ),
				'enqueue' => array(
					array( $this, 'maybe_enqueue_gf_tooltips_powertip_js' ),// front-end, any form with tooltips
				),
			),
			array(
				'handle'  => 'gf_tooltips_admin',
				'src'     => $this->get_base_url() . '/js/gf_tooltip_admin.js',
				'version' => $this->_version,
				'callback' => array( $this, 'localize_admin_scripts' ),
				'deps'    => array_merge(array('wp-color-picker'), $js_deps),
				'enqueue' => array(
					array( 'admin_page' => array('form_settings', 'plugin_settings', 'form_editor') )
				),
			),
		);

		return array_merge( parent::scripts(), $scripts );
	}

	public function localize_admin_scripts() {

		$params = array(
			'is_gf_min_2_5' => GFTT_GF_MIN_2_5,
		);
		wp_localize_script( 'gf_tooltips_admin', 'tooltipsVars', $params );

    }

	public function maybe_enqueue_gf_tooltips_powertip_js($form) {
	    return $this->form_has_tooltips($form);
	}

	/**
	 * Inject Javascript into the Gravity form editor page.
	 */
	public function editor_js() {

		?>
		<script type='text/javascript'>
			(function($){

				/* adding tooltip setting to all fields */
				$.each( window.fieldSettings, function(i, v){
					window.fieldSettings[i] += ", .gftt-tooltip";
				});

				$( document ).bind('gform_load_field_settings', function( event, field, form ) {

					if ( typeof field.jbTooltipContent === 'undefined' ) {
						field.jbTooltipContent = '';
					}

					$( '#gftt-tooltip' ).val( field.jbTooltipContent );

				});

				$( document ).ready(function(){

					var gfttIcon = '<i class="gftt-icon"></i>';

					$( document ).on('change', '#gftt-tooltip', function() {
						var $self = $( this );
						window.SetFieldProperty( 'jbTooltipContent', $self.val() );
						window.RefreshSelectedFieldPreview( function() {
							if ( $self.val() !== '' )
								$self.closest( '.gfield' ).find( '.gfield_label' ).append( gfttIcon );
						});
					});

				});

			})(jQuery);
		</script>
		<?php

	}

	/**
	 * Return the stylesheets which should be enqueued.
	 *
	 * @return array
	 */
	public function styles() {

		$styles = array(
			array(
				'handle'  => 'preload_gf_tooltips_font_woff2',
				'src'     => $this->get_base_url() . '/font/gfflt.woff2',
				'version' => '1590114356623',
				'enqueue' => array(
					array( $this, 'maybe_preload_assets' )
                )
			),
			array(
				'handle'  => 'preload_gf_tooltips_font_woff',
				'src'     => $this->get_base_url() . '/font/gfflt.woff',
				'version' => '1590114356623',
				'enqueue' => array(
					array( $this, 'maybe_preload_assets' )
                )
			),
			array(
				'handle'  => 'preload_gf_tooltips_font_ttf',
				'src'     => $this->get_base_url() . '/font/gfflt.ttf',
				'version' => '1590114356623',
				'enqueue' => array(
					array( $this, 'maybe_preload_assets' )
				)
			),
			array(
				'handle'  => 'gf_tooltips_font',
				'src'     => $this->get_base_url() . '/css/gf_tooltips_font.css',
				'version' => $this->_version,
				'enqueue' => array(
					array( 'admin_page' => array( 'form_editor', 'form_settings', 'plugin_settings' ) ),
					array( $this, 'maybe_enqueue_gf_tooltips_css' ),// front-end, any form with tooltips
				),
			),
			array(
				'handle'  => 'gf_tooltips',
				'src'     => $this->get_base_url() . '/css/gf_tooltips.css',
				'version' => $this->_version,
				'enqueue' => array(
					array( 'admin_page' => array( 'form_editor', 'form_settings', 'plugin_settings' ) ),
					array( $this, 'maybe_enqueue_gf_tooltips_css' ),// front-end, any form with tooltips
				),
			),
			array(
				'handle' => 'wp-color-picker',
				'enqueue' => array(
					array( 'admin_page' => array('form_settings', 'plugin_settings') )
				),
			),
		);

		return array_merge( parent::styles(), $styles );
	}

	public function maybe_preload_assets() {
	    return true;
	}

	public function configure_preload_assets( $html, $handle ) {

		if ( $handle === 'preload_gf_tooltips_font_woff2' ) {
			return str_replace( ["rel='stylesheet'", " media='all'"], ["rel='prefetch' as='font' type='font/woff2' crossorigin='anonymous'", ""], $html );
		}

		if ( $handle === 'preload_gf_tooltips_font_woff' ) {
			return str_replace( ["rel='stylesheet'", " media='all'"], ["rel='prefetch' as='font' type='font/woff' crossorigin='anonymous'", ""], $html );
		}

		if ( $handle === 'preload_gf_tooltips_font_ttf' ) {
			return str_replace( ["rel='stylesheet'", " media='all'"], ["rel='prefetch' as='font' type='font/ttf' crossorigin='anonymous'", ""], $html );
		}

		if ( $handle === 'preload_gf_tooltips_font' ) {
			return str_replace( ["rel='stylesheet'", " media='all'"], ["rel='prefetch' as='style' type='text/css' crossorigin='anonymous'", ""], $html );
		}

		return $html;
	}

	public function maybe_enqueue_gf_tooltips_css($form) {
		return $this->form_has_tooltips($form);
	}

	public function admin_inline_styles($hook) {
	    if ( $this->is_form_editor() ) {
			$form = $this->get_current_form();
			$this->add_inline_style($form);
		}
	}

	public function frontend_inline_styles( $form ) {
		$this->add_inline_style($form);
	}

	public function add_inline_style($form) {

		$has_tooltips = false;

		if ( !empty($form) && isset($form['fields']) ) {
			foreach ( $form['fields'] as &$field ) {
				if ( property_exists($field, 'jbTooltipContent') && !empty( $field->jbTooltipContent ) ) {
					$has_tooltips = true;
					break;
				}
			}
		}

		if ( !$has_tooltips ) {
			return;
		}

		$global_settings = $this->get_plugin_settings();
		if ( empty($global_settings) ) {
		    return;
		}

		$form_settings = $this->get_form_settings( $form );

		$settings = array();
		foreach($global_settings as $key => $val) {
			$settings[$key] = (!empty( $form_settings[$key] )) ? $form_settings[$key] : $val;
		}

		if ( !empty( $settings )) {
			$tooltip_styles = $this->get_inline_styles( $form, $settings );
			if ( !empty( $tooltip_styles ) ) {
				wp_register_style( 'gf_tooltips_inline', false );
				wp_enqueue_style( 'gf_tooltips_inline' );
				wp_add_inline_style( 'gf_tooltips_inline', $tooltip_styles );
			}
		}

    }

	public function get_inline_styles($form, $settings) {

	    $form_id = $form['id'];

		$icon = $settings['ttIcon'];
		$icon_sz = $settings['ttIconSize'];
		$icon_cl = $settings['ttIconColor'];
		$tip_cl = $settings['ttTipColor'];
		$tip_bg = $settings['ttTipBackground'];
		$tip_sz = $settings['ttTipSize'];
		$tip_wd = $settings['ttMaxWidth'];

		$css = '';
		$tip_css = '';
		$icon_css = '';

		if ( $tip_cl != '#333333' )
			$tip_css .= sprintf( 'color:%s;', $tip_cl );

		if ( $tip_bg != '#333333' )
			$tip_css .= sprintf( 'border-color:%1$s;background-color:%1$s;', $tip_bg );

		if ( $tip_sz != 16 )
			$tip_css .= sprintf( 'font-size:%spx;', $tip_sz );

		if ( $tip_wd != 300 )
			$tip_css .= sprintf( 'max-width:%spx;', $tip_wd );

		if ( !empty( $tip_css ) )
			$css .= sprintf( '#powerTip[class^="gftt-%s_"]{%s}', $form_id, $tip_css );


		if ( $icon_cl != '#333333' )
			$icon_css .= sprintf( 'color:%s;', $icon_cl );

		if ( $icon_sz != 18 )
			$icon_css .= sprintf( 'font-size:%spx;', $icon_sz );

		if ( !empty( $icon_css ) )
			$css .= sprintf( '#gform_wrapper_%s .gftt-icon{%s}', $form_id, $icon_css );


		if ( $icon != '\e803' )
			$css .= sprintf( '#gform_wrapper_%s .gftt-icon:before{content:"%s";}', $form_id, $icon );

		return trim( preg_replace( '/\s\s+/', '', $css ) );

	}

	public function form_has_tooltips($form) {

		$has_tooltips = false;

		if ( !empty($form) && isset($form['fields']) ) {
			foreach ( $form['fields'] as &$field ) {
				if ( property_exists($field, 'jbTooltipContent') && !empty( $field->jbTooltipContent ) ) {
					$has_tooltips = true;
					break;
				}
			}
        }

		return $has_tooltips;

    }

	// # SETTINGS -----------------------------------------------------------------------------------------------

    public function get_tooltip_settings_fields($include_default_values = true) {

	    $fields = array(
		    array(
			    'type'       => 'radio',
			    'name'       => 'ttIcon',
			    'label'      => esc_html__( 'Icon', 'gf_tooltips' ),
			    'tooltip'    => esc_html__( 'Select the icon that will be used on the tooltip.', 'gf_tooltips' ),
			    'choices'    => array(
				    array(
					    'label' => esc_html__( 'Icon 1', 'gf_tooltips' ),
					    'value' => '\e803'
				    ),
				    array(
					    'label' => esc_html__( 'Icon 2', 'gf_tooltips' ),
					    'value' => '\e806'
				    ),
				    array(
					    'label' => esc_html__( 'Icon 3', 'gf_tooltips' ),
					    'value' => '\f29c'
				    ),
				    array(
					    'label' => esc_html__( 'Icon 4', 'gf_tooltips' ),
					    'value' => '\e804'
				    ),
				    array(
					    'label' => esc_html__( 'Icon 5', 'gf_tooltips' ),
					    'value' => '\e801'
				    ),
				    array(
					    'label' => esc_html__( 'Icon 6', 'gf_tooltips' ),
					    'value' => '\e802'
				    ),
				    array(
					    'label' => esc_html__( 'Icon 7', 'gf_tooltips' ),
					    'value' => '\e805'
				    ),
				    array(
					    'label' => esc_html__( 'Icon 8', 'gf_tooltips' ),
					    'value' => '\f129'
				    )
			    )
		    ),
		    array(
			    'type'    => 'text',
			    'value'   => ($include_default_values) ? 18 : '',
			    'name'    => 'ttIconSize',
			    'class'	  => 'gftt-number',
			    'label'   => esc_html__( 'Icon Size', 'gf_tooltips' ),
			    'tooltip' => esc_html__( 'Enter the size of the tooltip icon.', 'gf_tooltips' )
		    ),
		    array(
			    'type'    => 'text',
			    'value'	  => ($include_default_values) ? '#333333' : '',
			    'data-default-color' => ($include_default_values) ? '#333333' : '',
			    'name'    => 'ttIconColor',
			    'class'	  => 'gftt-color-picker',
			    'label'   => esc_html__( 'Icon Color', 'gf_tooltips' ),
			    'tooltip' => esc_html__( 'Select the color that will be applied to the tooltip icon.', 'gf_tooltips' )
		    ),
		    array(
			    'type'    => 'text',
			    'value'   => ($include_default_values) ? 16 : '',
			    'name'    => 'ttTipSize',
			    'class'	  => 'gftt-number',
			    'label'   => esc_html__( 'Content Font Size', 'gf_tooltips' ),
			    'tooltip' => esc_html__( 'Enter the font size that will be applied to the tooltip text content.', 'gf_tooltips' )
		    ),
		    array(
			    'type'    => 'text',
			    'value'	  => ($include_default_values) ? '#ffffff' : '',
			    'data-default-color' => ($include_default_values) ? '#ffffff' : '',
			    'name'    => 'ttTipColor',
			    'class'	  => 'gftt-color-picker',
			    'label'   => esc_html__( 'Content Text Color', 'gf_tooltips' ),
			    'tooltip' => esc_html__( 'Select the color that will be applied to the tooltip text content.', 'gf_tooltips' )
		    ),
		    array(
			    'type'    => 'text',
			    'value'	  => ($include_default_values) ? '#333333' : '',
			    'data-default-color' => ($include_default_values) ? '#333333' : '',
			    'name'    => 'ttTipBackground',
			    'class'	  => 'gftt-color-picker',
			    'label'   => esc_html__( 'Content Wrap Background', 'gf_tooltips' ),
			    'tooltip' => esc_html__( 'Select the background color that will be applied to the tooltip content wrapper.', 'gf_tooltips' )
		    ),
		    array(
			    'type'       => 'select',
			    'name'       => 'ttPlacement',
			    'label'      => esc_html__( 'Placement', 'gf_tooltips' ),
			    'tooltip'    => esc_html__( 'Select the default placement of the tooltip when it is open relative to the tooltip icon. Placement can be top, right, bottom, left and many more. Please note that if a tooltip would extend outside of the viewport then its placement will be changed to an orientation that would be entirely within the current viewport.', 'gf_tooltips' ),
			    'choices'    => array(
				    array(
					    'label' => esc_html__( 'Select a position', 'gf_tooltips' ),
					    'value' => ''
				    ),
				    array(
					    'label' => esc_html__( 'Top Left Alt', 'gf_tooltips' ),
					    'value' => 'nw-alt'
				    ),
				    array(
					    'label' => esc_html__( 'Top Left', 'gf_tooltips' ),
					    'value' => 'nw'
				    ),
				    array(
					    'label' => esc_html__( 'Top', 'gf_tooltips' ),
					    'value' => 'n'
				    ),
				    array(
					    'label' => esc_html__( 'Top Right', 'gf_tooltips' ),
					    'value' => 'ne'
				    ),
				    array(
					    'label' => esc_html__( 'Top Right Alt', 'gf_tooltips' ),
					    'value' => 'ne-alt'
				    ),
				    array(
					    'label' => esc_html__( 'Right', 'gf_tooltips' ),
					    'value' => 'e'
				    ),
				    array(
					    'label' => esc_html__( 'Bottom Right Alt', 'gf_tooltips' ),
					    'value' => 'se-alt'
				    ),
				    array(
					    'label' => esc_html__( 'Bottom Right', 'gf_tooltips' ),
					    'value' => 'se'
				    ),
				    array(
					    'label' => esc_html__( 'Bottom', 'gf_tooltips' ),
					    'value' => 's'
				    ),
				    array(
					    'label' => esc_html__( 'Bottom Left', 'gf_tooltips' ),
					    'value' => 'sw'
				    ),
				    array(
					    'label' => esc_html__( 'Bottom Left Alt', 'gf_tooltips' ),
					    'value' => 'sw-alt'
				    ),
				    array(
					    'label' => esc_html__( 'Left', 'gf_tooltips' ),
					    'value' => 'w'
				    )
			    )
		    ),
		    array(
			    'type'    => 'text',
			    'value'   => ($include_default_values) ? 300 : '',
			    'name'    => 'ttMaxWidth',
			    'class'	  => 'gftt-width',
			    'label'   => esc_html__( 'Max Width', 'gf_tooltips' ),
			    'tooltip' => esc_html__( 'Enter the maximum width of the tooltip content wrapper.', 'gf_tooltips' )
		    )
	    );

	    return $fields;

    }

	/**
	 * Creates a settings page for this add-on.
	 */
	public function plugin_settings_fields() {

		$license = $this->get_plugin_setting('gf_tooltips_license_key');
		$status = get_option('gf_tooltips_license_status');

		$license_field = array(
			'name' => 'gf_tooltips_license_key',
			'tooltip' => esc_html__('Enter the license key you received after purchasing the plugin.', 'gf_tooltips'),
			'label' => esc_html__('License Key', 'gf_tooltips'),
			'type' => 'text',
			'input_type' => 'password',
			'class' => 'medium',
			'default_value' => '',
			'validation_callback' => array($this, 'license_validation'),
			'feedback_callback' => array($this, 'license_feedback'),
			'error_message' => esc_html__( 'Invalid license', 'gf_tooltips' ),
		);

		if (!empty($license) && !empty($status)) {
			$license_field['after_input'] = ($status == 'valid') ? ' License is valid' : ' Invalid or expired license';
		}

		$tooltip_settings_fields = $this->get_tooltip_settings_fields();

		$fields = array(
			array(
				'title'  => esc_html__('To unlock plugin updates, please enter your license key below', 'gf_tooltips'),
				'fields' => array(
					$license_field
				)
			),
			array(
				'title' => __( 'Tooltip Styles', 'gf_tooltips' ),
				'fields' => $tooltip_settings_fields,
			)
		);

		return $fields;
	}


	/**
	 * Form tooltips settings.
	 *
	 * @return array
	 */
	public function form_settings_fields( $form ) {

		$settings = $this->get_form_settings( $form );
	    $tooltip_settings_fields = $this->get_tooltip_settings_fields(false);

		$form_override_global_settings_value = (isset($settings['gf_tooltips_override_global']) && $settings['gf_tooltips_override_global'] == 1) ? 1 : 0;
		$form_override_global_settings_field = array(
			'name' => 'gf_tooltips_override_global',
			'label' => '',
			'type' => 'checkbox',
			'choices' => array(
				array(
					'label' => esc_html__('Override global tooltip options for this form?', 'gf_tooltips'),
					'tooltip' => esc_html__('If checked, you can override global tooltip styles for this form.', 'gf_tooltips'),
					'name' => 'gf_tooltips_override_global'
				)
			)
		);
		if (!empty($form_override_global_settings_value)) {
			$form_override_global_settings_field['choices'][0]['default_value'] = 1;
		}

		return array(
			array(
				'title'  => esc_html__( 'Tooltip Form Settings', 'gf_tooltips' ),
				'fields' => array($form_override_global_settings_field),
			),
            array(
                'title' => esc_html__( 'Any options that are left blank will inherit from the global styles in the main settings', 'gf_tooltips' ),
                'fields' => $tooltip_settings_fields,
            )
		);

	}

	// # ADMIN FUNCTIONS -----------------------------------------------------------------------------------------------

    public function custom_settings_tab( $tabs ) {
	    $tabs[] = array(
		    'id' => 'gftt', // Tab id is used later with the action hook that will output content for this specific tab.
		    'title' => esc_html__('Tooltip', 'gf_tooltips'),
		    'toggle_classes' => '', // Goes into the tab button class attribute.
		    'body_classes'   => '', // Goes into the tab content ( ul tag ) class attribute.
	    );

	    return $tabs;
    }

    public function custom_settings_markup( $form ) {
	    ?>

        <li class="gftt-tooltip field_setting">
            <label for="gftt-tooltip" class="section_label">
			    <?php esc_html_e( 'Tooltip Content', 'gf_tooltips' ); ?>
			    <?php gform_tooltip( 'gftt-tooltip' ) ?>
            </label>
            <textarea id="gftt-tooltip" class="fieldwidth-3"></textarea>
        </li>

	    <?php
    }

	/**
	 * Add additional options to the field general settings.
	 */
	public function field_general_settings( $position, $form_id ) {

		if ( $position == 10 ) {
			$this->custom_settings_markup( GFAPI::get_form( $form_id ) );
		}

	}


	/**
	 * Add tooltips option on field appearance settings.
	 */
	public function field_tooltips( $tooltips ) {

		$tooltips['gftt-tooltip'] = sprintf(
			'<h6>%s</h6>%s',
			esc_html__( 'Tooltip Content', 'gf_tooltips' ),
			esc_html__( 'This will be the content that appears on the tooltip.', 'gf_tooltips' )
		);

		return $tooltips;

	}


	// # DISPLAY -----------------------------------------------------------------------------------------------

	/**
	 * This will modify the way the field content is rendered.
	 */
	public function field_content( $content, $field, $value, $lead_id, $form_id ) {

		$tip_content = ( property_exists($field, 'jbTooltipContent') && !empty( $field->jbTooltipContent ) ) ? trim( __( $field->jbTooltipContent, 'gf_tooltips'.'-'.$form_id ) ) : "";
		if ( empty($tip_content) || empty($content) ) {
		    return $content;
		}

		$tip_id = 'gftt-' . $form_id . '_' . $field->id;

		if ( ! class_exists( 'JSLikeHTMLElement' ) ) {
			// load our custom updater if it doesn't already exist
			include( dirname( __FILE__ ) . '/inc/JSLikeHTMLElement.php' );
		}

		$dom = new DOMDocument;

		// keeping these in, but they seem to do nothing
		$dom->preserveWhitespace = true;
		$dom->formatOutput = false;

		// set error level
		$internalErrors = libxml_use_internal_errors(true);

		$dom->registerNodeClass('DOMElement', 'JSLikeHTMLElement');
		//$dom->loadHTML(mb_convert_encoding("<section>".$content."</section>", 'HTML-ENTITIES', 'UTF-8'), LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD);
        // XML_PARSE_NOBLANKS
		$dom->loadHTML(mb_convert_encoding($content, 'HTML-ENTITIES', 'UTF-8'));

		// Restore error level
		libxml_use_internal_errors($internalErrors);

		$selector = new DOMXPath($dom);
		$label_elems = $selector->query('//*[contains(@class, "gfield_label")]');// first try normal field label

		if ($label_elems->length == 0) {
			$label_elems = $selector->query('//*[contains(@class, "gsection_title")]');// then try section title
		}

		if ($label_elems->length == 0) {
		    return $content;
		}

		$label = $label_elems->item(0);

		$label_content = ( property_exists($field, 'isRequired') && $field->isRequired ) ? $field->label . "<span class='gfield_required'>*</span>" : $field->label;


		if ( $field->is_form_editor() ) {

			$tip_content = sprintf('<span class="gftt-label">%s</span><i class="gftt-icon"></i>', $label_content);

		}
		else {

			$tip_placement = 'nw-alt';// default placement
			$form = GFAPI::get_form( $form_id );

			$global_settings = $this->get_plugin_settings();
			$form_settings = $this->get_form_settings( $form );

			if (!empty($global_settings) && isset($global_settings['ttPlacement']) && !empty($global_settings['ttPlacement'])) {
				$tip_placement = $global_settings['ttPlacement'];
			}
			if (!empty($form_settings) && isset($form_settings['ttPlacement']) && !empty($form_settings['ttPlacement'])) {
				$tip_placement = $form_settings['ttPlacement'];
			}

			$tip_content = sprintf(
				'<span class="gftt-label">%3$s</span><i id="%1$s" class="gftt-icon" data-placement="%4$s"></i><span class="gftt-content"><span id="%1$s-content" data-tid="%1$s">%2$s</span></span>',
				$tip_id,
				$tip_content,
				$label_content,
				$tip_placement
			);

		}

		$label->innerHTML = do_shortcode( $tip_content );

		$html = $dom->saveHTML($dom->documentElement);

		if ( $field->type == 'radio' || $field->type == 'checkbox' || $field->inputType == 'radio' || $field->inputType == 'checkbox' ) {
			// put whitespace back in between label and radio/checkbox inputs
			return str_replace( "><label ", ">\n<label ", $html );
		}

		return $html;
		//return utf8_decode( $dom->saveHTML($dom->documentElement) );

	}


	// # UPDATES -----------------------------------------------------------------------------------------------

	/**
	 * Add custom messages after plugin row based on license status
	 */

	public function gf_plugin_row($plugin_file='', $plugin_data=array(), $status='') {
		$row = array();
		$license_key = trim($this->get_plugin_setting('gf_tooltips_license_key'));
		$license_status = get_option('gf_tooltips_license_status', '');
		if (empty($license_key) || empty($license_status)) {
			$row = array(
				'<tr class="plugin-update-tr">',
					'<td colspan="3" class="plugin-update gf_tooltips-plugin-update">',
						'<div class="update-message">',
							'<a href="' . admin_url('admin.php?page=gf_settings&subview=' . $this->_slug) . '">Activate</a> your license to receive plugin updates and support. Need a license key? <a href="' . $this->_url . '" target="_blank">Purchase one now</a>.',
						'</div>',
						'<style type="text/css">',
							'.plugin-update.gf_tooltips-plugin-update .update-message:before {',
								'content: "\f348";',
								'margin-top: 0;',
								'font-family: dashicons;',
								'font-size: 20px;',
								'position: relative;',
								'top: 5px;',
								'color: orange;',
								'margin-right: 8px;',
							'}',
							'.plugin-update.gf_tooltips-plugin-update {',
								'background-color: #fff6e5;',
							'}',
							'.plugin-update.gf_tooltips-plugin-update .update-message {',
								'margin: 0 20px 6px 40px !important;',
								'line-height: 28px;',
							'}',
						'</style>',
					'</td>',
				'</tr>'
			);
		}
		elseif(!empty($license_key) && $license_status != 'valid') {
			$row = array(
				'<tr class="plugin-update-tr">',
					'<td colspan="3" class="plugin-update gf_tooltips-plugin-update">',
						'<div class="update-message">',
							'Your license is invalid or expired. <a href="'.admin_url('admin.php?page=gf_settings&subview='.$this->_slug).'">Enter valid license key</a> or <a href="'.$this->_url.'" target="_blank">purchase a new one</a>.',
							'<style type="text/css">',
								'.plugin-update.gf_tooltips-plugin-update .update-message:before {',
									'content: "\f348";',
									'margin-top: 0;',
									'font-family: dashicons;',
									'font-size: 20px;',
									'position: relative;',
									'top: 5px;',
									'color: #d54e21;',
									'margin-right: 8px;',
								'}',
								'.plugin-update.gf_tooltips-plugin-update {',
									'background-color: #ffe5e5;',
								'}',
								'.plugin-update.gf_tooltips-plugin-update .update-message {',
									'margin: 0 20px 6px 40px !important;',
									'line-height: 28px;',
								'}',
							'</style>',
						'</div>',
					'</td>',
				'</tr>'
			);
		}

		echo implode('', $row);
	}


	// # LICENSING -----------------------------------------------------------------------------------------------

	/**
	 * Determine if the license key is valid so the appropriate icon can be displayed next to the field.
	 *
	 * @param string $value The current value of the license_key field.
	 * @param array $field The field properties.
	 *
	 * @return bool|null
	 */
	public function license_feedback( $value, $field ) {
		if ( empty( $value ) ) {
			return null;
		}

		// Send the remote request to check the license is valid
		$license_data = $this->perform_edd_license_request( 'check_license', $value );

		$valid = null;
		if ( empty( $license_data ) || !is_object($license_data) || !property_exists($license_data, 'license') || $license_data->license == 'invalid' ) {
			$valid = false;
		}
		elseif ( $license_data->license == 'valid' ) {
			$valid = true;
		}

		if (!empty($license_data) && is_object($license_data) && property_exists($license_data, 'license')) {
			update_option('gf_tooltips_license_status', $license_data->license);
		}

		return $valid;
	}


	/**
	 * Handle license key activation or deactivation.
	 *
	 * @param array $field The field properties.
	 * @param string $field_setting The submitted value of the license_key field.
	 */
	public function license_validation( $field, $field_setting ) {
		$old_license = $this->get_plugin_setting( 'gf_tooltips_license_key' );

		if ( $old_license && $field_setting != $old_license ) {
			// Send the remote request to deactivate the old license
			$response = $this->perform_edd_license_request( 'deactivate_license', $old_license );
			if ( !empty($response) && is_object($response) && property_exists($response, 'license') && $response->license == 'deactivated' ) {
				delete_option('gf_tooltips_license_status');
			}
		}

		if ( ! empty( $field_setting ) ) {
			// Send the remote request to activate the new license
			$response = $this->perform_edd_license_request( 'activate_license', $field_setting );
			if ( !empty($response) && is_object($response) && property_exists($response, 'license') ) {
				update_option('gf_tooltips_license_status', $response->license);
			}
		}
	}


	/**
	 * Send a request to the EDD store url.
	 *
	 * @param string $edd_action The action to perform (check_license, activate_license or deactivate_license).
	 * @param string $license The license key.
	 *
	 * @return object
	 */
	public function perform_edd_license_request( $edd_action, $license ) {

		// Prepare the request arguments
		$args = array(
			'timeout' => GFTT_TIMEOUT,
			'sslverify' => GFTT_SSL_VERIFY,
			'body' => array(
				'edd_action' => $edd_action,
				'license' => trim($license),
				'item_name' => urlencode(GFTT_NAME),
				'url' => home_url(),
			)
		);

		// Send the remote request
		$response = wp_remote_post(GFTT_HOME, $args);

		return json_decode( wp_remote_retrieve_body( $response ) );
	}


	public function debug_output($data = '', $background='black', $color='white') {
		echo '<pre style="padding:20px; background:'.$background.'; color:'.$color.';">';
			print_r($data);
		echo '</pre>';
	}



} // end class
