<?php

/**
 * Manage all the front end code for Cornerstone
 * including shortcode styling and scripts
 */

class Cornerstone_Front_End extends Cornerstone_Plugin_Component {

	public $dependencies = array( 'Inline_Scripts' );

	/**
	 * Setup hooks
	 */
	public function setup() {

		add_filter('template_include', array( $this, 'setup_after_template_include' ), 99998 );

		// Enqueue Scripts & Styles

		add_action( 'wp_enqueue_scripts', array( $this, 'scripts' ), 5 );
		add_action( 'wp_enqueue_scripts', array( $this, 'styles' ), 5 );
		add_action( 'cs_late_template_redirect', array( $this, 'postLoaded' ), 9998, 0 );

		// Excerpt related functions
		add_filter( 'strip_shortcodes_tagnames', array($this, 'preserve_excerpt'), 999999 );

		add_filter( 'the_content', array( $this, 'maybe_noemptyp' ), 10 );

		add_shortcode( 'cs_content_seo', array( $this, 'cs_content_seo' ) );

    add_action( 'wp_head', array( $this, 'cs_head_late'), 10000 );
    add_action( 'wp_head', array( $this, 'cs_head_late_after'), 10001 );

    add_action( 'cs_element_rendering', array( $this, 'register_scripts') );

    add_filter( 'script_loader_tag', array( $this, 'fix_script_tags'), 0, 3 );

	}

	/**
	 * A late template_redirect hook allows plugins like Custom 404 and Under Construction
	 * to modify the query before we assume we can query info like the current ID
	 */
	public function setup_after_template_include( $template ) {
		do_action('cs_late_template_redirect');
		return $template;
	}

	/**
	 * Enqueue Styles
	 */
	public function styles() {

		$settings = CS()->settings();
    if ( isset( $settings['enable_legacy_font_classes'] ) && $settings['enable_legacy_font_classes'] ) {
			$fa_icons_asset = $this->plugin->css( 'site/fa-icon-classes' );
			wp_enqueue_style( 'x-fa-icon-classes', $fa_icons_asset['url'], array(), $fa_icons_asset['version'] );
		}

	}

	/**
	 * Enqueue Scripts
	 */
	public function scripts() {
  	$this->register_scripts();
  	wp_enqueue_script( 'cornerstone-site-body' );
	}

  public function register_scripts() {
		$script_asset = CS()->js( 'site/cs' );
		wp_register_script( 'cornerstone-site-body', $script_asset['url'], array( 'jquery' ), $script_asset['version'], true );
		wp_localize_script( 'cornerstone-site-body', 'csJsData', [
			'linkSelector' => apply_filters( 'cs_link_selector', '#cs-content a[href*="#"]' )
		]);
  	wp_register_script( 'cs-ilightbox', $this->url( 'assets/js/site/ilightbox.js' ), array( 'jquery' ), CS_VERSION, true );
		wp_register_script( 'cs-flexslider', $this->url( 'assets/js/site/flexslider.js' ), array( 'jquery' ), CS_VERSION, true );
  }

	public function postLoaded() {
		CS()->novus()->service('Styling')->addStyles( 'fa-config', $this->view( 'frontend/font-awesome', false, $this->plugin->common()->get_fa_config(), true), 5 );
	}

	/**
	 * Preserve content of [cs_content_seo][/cs_content_seo] making it visible for excerpt generation.
	 */
	public function preserve_excerpt ( $tags ) {
		return array_diff ($tags, array('cs_content_seo'));
	}

	/**
	 * Cornerstone adds a wrapping [cs_content] shortcode.Run the content through
	 * cs_noemptyp if we know it was originally generated by Cornerstone.
	 * This cleans up any empty <p> tags.
	 * @param  string $content Early the_content. Before do_shortcode
	 * @return string          the_content with empty <p> tags removed and wrapping div
	 */
	public function maybe_noemptyp( $content ) {

		if ( false !== strpos( $content, '[cs_content]' ) && false !== strpos( $content, '[/cs_content]' ) ) {
			$content = cs_noemptyp( $content );
		}

		return $content;

	}

	public function cs_content_seo ($atts, $content) {

		extract( shortcode_atts( array(
			'output'      => false
		), $atts, 'cs_content_seo' ) );

		if ( $output || doing_filter ('get_the_excerpt') ) return $content;

		return '';

	}

	public function output_layout_content( $content ) {
    if ( is_scalar( $content ) ) {
      echo $content;
    }
	}

  public function cs_head_late() {
    do_action( 'cs_head_late' );
  }

  public function cs_head_late_after() {
    do_action( 'cs_head_late_after' );
  }

  public function fix_script_tags( $tag, $handle, $src ) {
    return $this->plugin->component('Common')->fix_script_tags( $handle, $tag );
	}

}
