<?php

namespace Itinerator\Templates;

use DateTime;
use DateTimeZone;

class Events extends BlockTemplate
{
  protected $template_slug = 'events';
  protected $block_type = 'events';
  protected $block_attributes = "events";
  public $query_key = "region_id";
  public $events;
  public $matchRecordsByTagCategory;
  public $buttonText;

  public static function getRecurringEventOccurrences($eventTimes, $limit = 3)
  {
    $occurrences = [];
    $now = new DateTime('now', self::getWpTimezone());

    // Collect all upcoming times from all time slots
    $allUpcomingTimes = [];
    foreach ($eventTimes as $timeSlot) {
      if (!empty($timeSlot->upcoming_times)) {
        foreach ($timeSlot->upcoming_times as $time) {
          $startDate = self::convertToWpTimezone($time->start);
          if ($startDate > $now) {
            // Include the full time slot data to access locations
            $allUpcomingTimes[] = [
              'time' => $time,
              'locations' => $timeSlot->locations ?? [],
              'all_day' => $timeSlot->all_day ?? false
            ];
          }
        }
      }
    }

    // Sort all times by start date
    usort($allUpcomingTimes, function ($a, $b) {
      return strtotime($a['time']->start) - strtotime($b['time']->start);
    });

    // Get the next occurrences
    foreach (array_slice($allUpcomingTimes, 0, $limit) as $occurrence) {
      $datetimes = self::convertDateTimeRange($occurrence['time']->start, $occurrence['time']->end);
      $occurrences[] = [
        'start' => $datetimes['date_range'],
        'start_raw' => $occurrence['time']->start,
        'time' => !$occurrence['all_day'] ? $datetimes['time_range'] : 'All Day',
        'locations' => $occurrence['locations']
      ];
    }

    return $occurrences;
  }

  public static function isRecurring($event)
  {
    if (empty($event)) {
        return false;
    }

    // Convert to object if it's an array
    $event = is_array($event) ? (object)$event : $event;

    return !empty($event->recurring) && $event->recurring === true;
  }

  public static function isSeries($event)
  {
    if (empty($event)) {
        return false;
    }

    // Convert to object if it's an array
    $event = is_array($event) ? (object)$event : $event;

    // Check if the event itself is recurring
    if (!empty($event->recurring) && $event->recurring === true) {
        return true;
    }

    // Check if there are multiple event times
    if (!empty($event->event_times) && is_array($event->event_times) && count($event->event_times) > 1) {
        return true;
    }

    // Check if any individual event_time is recurring
    if (!empty($event->event_times) && is_array($event->event_times)) {
        foreach ($event->event_times as $event_time) {
            if (!empty($event_time->recurring) && $event_time->recurring === true) {
                return true;
            }
        }
    }

    return false;
  }

  public static function getWpTimezone()
  {
    static $timezone = null;
    if ($timezone === null) {
      $timezone = wp_timezone();
    }
    return $timezone;
  }

  public static function convertToWpTimezone($dateTime)
  {
    return (new DateTime($dateTime, new DateTimeZone('UTC')))->setTimezone(self::getWpTimezone());
  }

  public static function convertDateTimeRange($startDateTime, $endDateTime, $dateFormat = null, $timeFormat = null)
  {
    $startDateTimeObj = self::convertToWpTimezone($startDateTime);
    $endDateTimeObj = self::convertToWpTimezone($endDateTime);

    $currentYear = date('Y');

    // Use provided formats or fall back to defaults
    $dateFormat = $dateFormat ?? 'D, M j';
    $timeFormat = $timeFormat ?? 'g:i a';

    $formattedStartDate = $startDateTimeObj->format($dateFormat);
    $formattedEndDate = $endDateTimeObj->format($dateFormat);

    $formattedStartTime = $startDateTimeObj->format($timeFormat);
    $formattedEndTime = $endDateTimeObj->format($timeFormat);

    $isOutsideCurrentYear = ($startDateTimeObj->format('Y') !== $currentYear) || ($endDateTimeObj->format('Y') !== $currentYear);
    $isSingleDate = $formattedStartDate === $formattedEndDate;

    if ($isOutsideCurrentYear) {
        $formattedEndDate .= ', ' . $endDateTimeObj->format('Y');
    }

    $dateRange = $isSingleDate ? $formattedStartDate : ($formattedStartDate . ' - ' . $formattedEndDate);
    $timeRange = $formattedStartTime . ' - ' . $formattedEndTime;

    $wrappedDateRange = '<time datetime="' . $startDateTime . '">' . $dateRange . '</time>';
    $wrappedTimeRange = '<time>' . $timeRange . '</time>';

    return [
        'date_range' => $wrappedDateRange,
        'time_range' => $wrappedTimeRange,
        'next_start_day' => $startDateTimeObj->format('j'),
        'next_start_month' => $startDateTimeObj->format('M'),
        'is_single_date' => $isSingleDate
    ];
  }

  protected function get_data()
  {
    $params = $this->record_list_query();
    $data = query_itin_api('events?' . $params);

    $this->events = array_map(function ($event) {
      return $this->jm->map($event, new Event($this));
    }, $data);
  }

  public static function getNextEventTime($event_times)
  {
    if (empty($event_times)) {
        return null;
    }

    $now = new DateTime('now', self::getWpTimezone());
    $nearest = null;
    $nearestDiff = PHP_FLOAT_MAX;

    foreach ($event_times as $time) {
        // Skip if next_start or next_end is empty
        if (empty($time->next_start) || empty($time->next_end)) {
            continue;
        }

        try {
            $startDate = self::convertToWpTimezone($time->next_start);
            $endDate = self::convertToWpTimezone($time->next_end);

            // If the event is currently ongoing (started in past but ends in future)
            if ($startDate <= $now && $endDate >= $now) {
                return $time; // Return immediately as this is happening now
            }

            // For future events, calculate the difference
            $diff = $startDate->getTimestamp() - $now->getTimestamp();
            if ($diff >= 0 && ($diff < $nearestDiff || $nearest === null)) {
                $nearest = $time;
                $nearestDiff = $diff;
            }
        } catch (\Exception $e) {
            continue; // Skip invalid dates
        }
    }

    return $nearest;
  }
}
