<?php
if (!defined('ABSPATH')) {
  exit;
}


function query_itin_api($query_path, $attr = false, $query_param = [], $api_token = API_TOKEN)
{
    $cache_enabled = get_option('itinerator_cache_enabled', 'no');
    $cache_key = 'itin_api_' . md5($query_path . serialize($query_param));

    // Only check cache if caching is enabled
    if ($cache_enabled === 'yes') {
        $cached_data = get_transient($cache_key);
        if ($cached_data !== false) {
            return $cached_data;
        }
    }

    try {

        // Build URL and make request
        $url = add_query_arg($query_param, API_URL . $query_path);
        $response = wp_remote_get($url, [
            'headers' => [
                'Authorization' => 'Bearer ' . $api_token
            ],
            'timeout' => 30
        ]);

        // Check for errors
        if (is_wp_error($response)) {
            throw new \Exception("API request failed");
        }

        // Get response code
        $http_code = wp_remote_retrieve_response_code($response);

        // Decode JSON response
        $data = json_decode(wp_remote_retrieve_body($response));


        // Check for JSON decoding errors
        if (json_last_error() != JSON_ERROR_NONE) {
            throw new \Exception("Error parsing JSON response");
        }

        // Only cache if caching is enabled
        if ($cache_enabled === 'yes') {
            set_transient($cache_key, $data, 24 * HOUR_IN_SECONDS);
        }

        // Return data
        // return ['response' => $http_code, 'data' => $data];
        return $data;
    } catch (\Exception $e) {

        // Log error message
        error_log($e->getMessage());

        // Return default value
        //This indicates to the calling code that an error occurred.
        return [
            'error' => true,
            'message' => $e->getMessage()
        ];
        // some other options for the return:
        // return []; //If the calling code expects data, this signals no data was retrieved.
        // throw $e; //Let the exception bubble up to be handled elsewhere.
        // return; //Simpler approach without needing a return value.
    }
}

// Add the AJAX handler for the toggle
add_action('wp_ajax_toggle_itin_cache', 'toggle_itin_cache');
function toggle_itin_cache() {
    check_ajax_referer('toggle_itin_cache_nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(['message' => 'Insufficient permissions']);
        return;
    }

    $enabled = $_POST['enabled'] === 'yes' ? 'yes' : 'no';
    update_option('itinerator_cache_enabled', $enabled);

    // If cache is being disabled, clear existing cache
    if ($enabled === 'no') {
        global $wpdb;
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_itin_api_%'");
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_itin_api_%'");
    }

    wp_send_json_success(['message' => 'Cache setting updated']);
}

