import { InspectorControls, useBlockProps, RichText } from "@wordpress/block-editor";
import {
  Edit,
  LinkedPageSelect,
  OptionsPanel,
  TagFilters,
  ToggleControls,
  useEditContext,
} from "./common";
import attributes from "../attributes/challenges.json";
import { ItineratorBlock } from "./ItineratorBlock";
import { useItinPageMeta, useRecords } from "../hooks";
import { TagCategoryControl } from "./common/TagCategoryControl";
import { useItinSample } from "./store";

new ItineratorBlock("challenges", {
  attributes,
  supports: {
    align: true,
    customClassName: true,
  },
  edit: (props) => {
    const { itinPageType, itinPageId } = useItinPageMeta();
    const { sample, loading } = useItinSample(itinPageType, itinPageId);
    if (loading) return null;
    return (
      <Edit {...props}>
        <Sidebar />
        <Display sample={sample} />
      </Edit>
    );
  },
});

const Sidebar = () => {
  return (
    <InspectorControls key="settings">
      <OptionsPanel>
        <LinkedPageSelect type="challenge" />
        <ToggleControls />
        <TagCategoryControl />
      </OptionsPanel>
      <TagFilters />
    </InspectorControls>
  );
};

const Display = ({ sample }) => {
  const { attributes } = useEditContext();
  const { matchRecordsByTagCategory, showImage, showName, showDescription, showLocation, buttonText } = attributes;
  const { itinPageId, itinPageType } = useItinPageMeta();
  const id = itinPageId || (sample && sample.id.toString());

  const blockProps = useBlockProps({
    className: `itin itinerator-challenges ${attributes.align ? `align${attributes.align}` : ''} ${attributes.customClassName ? `${attributes.customClassName}` : ''}`,
  });
  const { records, loading } = useRecords(
    `challenges`,
    { region_id: itinPageType === 'region' && !matchRecordsByTagCategory ? id : null },
    id
  );
  return (
    <div {...blockProps}>
      <div className="container-fluid">
        {!records.length && !loading && "No Challenges found"}
        {records.map((challenge) => (
          <div
            key={challenge.id}
            className={`challenge-row row justify-content-end align-items-center searchable-item mb-4 ${challenge.slug} ${challenge.tags.map((t) => `tag-id-${t.tag_id}`).join(" ")}`}
          >
            <div className="col-12 col-md-6 order-md-2 p-0 open-half">
              {attributes.showImage && challenge.hero && (
                <div className="image">
                  <img src={challenge.hero.url} className="img-fluid" />
                </div>
              )}
              <a className="stretched-link"></a>
            </div>
            <div className="col-12 col-md-6 order-md-1 contained-half">
              <div className="content">
                {attributes.showName && challenge.name && <h2>{challenge.name}</h2>}
                {attributes.showLocation && challenge.target_name && <h3>{challenge.target_name}</h3>}
                {attributes.showDates && challenge.rewards.map((reward) => (
                  <div key={reward.id}>{reward.date_string}</div>
                ))}
                {attributes.showDescription && challenge.description && (
                  <div
                    dangerouslySetInnerHTML={{
                      __html: challenge.description,
                    }}
                  ></div>
                )}
                {attributes.showName && challenge.name && (
                  <a

                    className="btn btn-primary"
                    aria-label={`Explore ${challenge.name}`}
                  >
                    <RichText
                      tagName="span"
                      value={buttonText}
                      onChange={(t) => setAttributes({ buttonText: t })}
                    />{" "}
                    {challenge.name}
                  </a>
                )}
              </div>
            </div>
          </div>
        ))}
      </div>
    </div>
  );
}
