import { __ } from "@wordpress/i18n";
import { Fragment, useCallback } from "@wordpress/element";
import {
  Button,
  HorizontalRule,
  PanelBody,
  PanelRow,
  SelectControl,
} from "@wordpress/components";

import {
  useAfterMountEffect,
  useTagCategories,
  useTagsByCategoryOptions,
} from "../../hooks";
import { useEditContext } from "./Edit";

const TagFilter = ({ filter, index, options }) => {
  const { attributes, setAttributes } = useEditContext();
  const { tagFilters } = attributes;
  const { category, id } = filter;
  const { tagOptions, defaultOption } = useTagsByCategoryOptions(category);

  const handleTagCategoryChange = (cat) => {
    const newFilter = [...tagFilters];
    newFilter[index] = { category: cat, id: "" };
    setAttributes({ tagFilters: newFilter });
  };

  const handleTagNameChange = useCallback(
    (n) => {
      const newFilter = [...tagFilters];
      newFilter[index].id = n;
      setAttributes({ tagFilters: newFilter });
    },
    [tagFilters, index, setAttributes]
  );

  const handleRemoveFilter = () => {
    const newFilter = tagFilters.filter((t, i) => i !== index);
    setAttributes({ tagFilters: newFilter });
  };

  const hasValidOption =
    !tagOptions.length || !!tagOptions.find((t) =>t.value === id);

  useAfterMountEffect(() => {
    if (!hasValidOption) {
      handleTagNameChange(defaultOption);
    }
  }, [handleTagNameChange, hasValidOption, defaultOption]);

  return (
    <Fragment>
      <PanelRow>
        <SelectControl
          label="Tag Category"
          options={options}
          value={category}
          onChange={handleTagCategoryChange}
        />
      </PanelRow>
      <PanelRow>
        <SelectControl
          label="Tag Name"
          options={tagOptions}
          value={id}
          onChange={handleTagNameChange}
        />
      </PanelRow>
      <PanelRow>
        <Button
 variant="link"
          icon="remove"
          className="is-tertiary"
          onClick={handleRemoveFilter}
          isDestructive
          style={{ boxShadow: "none" }}
        >
          Remove filter
        </Button>
      </PanelRow>
      <HorizontalRule
        style={{ borderBottom: "#e0e0e0", margin: "0 0 0.5rem" }}
      />
    </Fragment>
  );
};

export const TagFilters = () => {
  const { attributes, setAttributes } = useEditContext();
  const { tagFilters } = attributes;
  const { defaultCategory, options } = useTagCategories();

  return (
    <PanelBody title="Tag Filters">
      {tagFilters.map((filter, i) => {
        return <TagFilter filter={filter} index={i} options={options} key={i} />;
      })}
      <PanelRow>
        <Button
          variant="link"
          icon="insert"
          className="is-tertiary"
          onClick={() =>
            setAttributes({
              tagFilters: tagFilters.concat({
                category: defaultCategory,
                id: "",
              }),
            })
          }
        >
          Add new filter
        </Button>
      </PanelRow>
    </PanelBody>
  );
};
