import { InfoBox, InfoWindow, useGoogleMap } from "@react-google-maps/api";
import { Fragment, useRef, useState, useEffect } from "@wordpress/element";
import { usePrevious } from "../../../hooks";
import { useEditContext } from "../Edit";
import { CustomMarker } from "./CustomMarker";
import { CustomPolygon } from "./CustomPolygon";

const getLabelPositions = (width, height) => {
  return new google.maps.Size(-(width / 2), height);
};

export const Location = ({
  type,
  location,
  position,
  showMarker,
  showBoundaries,
  showLabel,
  boundsRef,
  activeMarkerId,
  setActiveMarkerId,
  linkedPage,
  tilesLoaded,
}) => {
  const {
    attributes: { markerLabelOriginHeight, markerLabelClass },
  } = useEditContext();

  const { id, slug, name, short_description, address, hero, lat, lng } =
    location;

  const [opacity, setOpacity] = useState(showLabel ? 0 : 1);
  const [labelReady, setLabelReady] = useState(showLabel ? 0 : 1);
  const prevLabelReady = usePrevious(labelReady);
  const [labelWidth, setLabelWidth] = useState(150);
  const labelRef = useRef();
  const infoBoxRef = useRef();
  const map = useGoogleMap();
  const setBounds = () => {
    map.fitBounds(boundsRef.current, 0);
  };
  const onPolyLoad = (polygon) => {
    if (showBoundaries) {
      var paths = polygon.getPaths();
      var path;
      for (var i = 0; i < paths.getLength(); i++) {
        path = paths.getAt(i);
        for (var ii = 0; ii < path.getLength(); ii++) {
          boundsRef.current.extend(path.getAt(ii));
        }
      }
      setBounds();
    }
  };

  const onMarkerLoad = (marker) => {
    if (showMarker && !showBoundaries) {
      boundsRef.current.extend(marker.position);
      setBounds();
    }
  };

  const handleMarkerClick = () => {
    !!linkedPage && type !== "record" && setActiveMarkerId(location.id);
  };

  const handleBoundaryClick = () => {
    window.location.href = linkedPage.link + location.slug;
  };

  useEffect(() => {
    const incrementOpacity = (opacity) => {
      setTimeout(() => {
        const newOpacity = Math.round((opacity + 0.1) * 100) / 100;
        setOpacity(newOpacity);
        if (newOpacity < 1) {
          incrementOpacity(newOpacity);
        }
      }, 50);
    };
    if (labelReady && prevLabelReady !== labelReady) {
      incrementOpacity(opacity);
    }
  }, [labelReady, prevLabelReady, opacity]);

  useEffect(() => {
    if (infoBoxRef.current) infoBoxRef.current.draw();
  }, [labelWidth, markerLabelOriginHeight]);

  useEffect(() => {
    if (labelRef.current) {
      setLabelWidth(labelRef.current.offsetWidth);
      setLabelReady(true);
    }
  }, [tilesLoaded]);

  return (
    <Fragment>
      <CustomMarker
        position={position}
        visible={showMarker}
        location={location}
        onClick={handleMarkerClick}
        onLoad={onMarkerLoad}
        opacity={opacity}
        key={location.id}
      >
        {activeMarkerId === id && (
          <InfoWindow
            onCloseClick={setActiveMarkerId}
            options={{ zIndex: 100 }}
            key={`info` + id}
          >
            <div className="itinerator-info-window">
              {!!hero && <img src={hero} />}
              <div className="itinerator-info-window-content">
                <strong>{name}</strong>
                <p
                  dangerouslySetInnerHTML={{
                    __html: address || short_description,
                  }}
                />
                <div className="itinerator-info-window-links">
                  <strong>
                    <a href={linkedPage.link + slug}>Learn More</a>
                  </strong>
                  <a
                    href={`https://www.google.com/maps/dir//${lat},${lng}`}
                    target={"_blank"}
                  >
                    Get Directions
                    <span class="dashicons dashicons-external"></span>
                  </a>
                </div>
              </div>
            </div>
          </InfoWindow>
        )}
      </CustomMarker>
      <InfoBox
        position={position}
        options={{
          closeBoxURL: "",
          alignBottom: true,
          enableEventPropagation: true,
          pixelOffset: getLabelPositions(labelWidth, markerLabelOriginHeight),
          maxWidth: 0,
          visible: showLabel,
        }}
        onLoad={(ib) => {
          infoBoxRef.current = ib;
        }}
        key={`info2` + id}
      >
        <div
          ref={labelRef}
          className={markerLabelClass}
          onClick={(e) => {
            e.stopPropagation();
            handleMarkerClick();
          }}
          style={{ opacity: opacity }}

        >
          {name}
        </div>
      </InfoBox>
      <CustomPolygon
        path={location.geofence || []}
        visible={showBoundaries}
        onLoad={onPolyLoad}
        onClick={handleBoundaryClick}
        key={`poly` + id}
      />
    </Fragment>
  );
};
