import { useState, useEffect } from "react";
import { InspectorControls, useBlockProps } from "@wordpress/block-editor";
import { PanelRow, SelectControl, ToggleControl } from "@wordpress/components";
import { Fragment } from "@wordpress/element";
import { __ } from "@wordpress/i18n";
import { startCase } from "lodash";
import { Edit, OptionsPanel, ToggleControls, useEditContext } from "./common";
import { useItinPageMeta } from "../hooks";
import { SAMPLE_CONTENT } from "../sample-data";
import attributes from "../attributes/content.json";
import { ItineratorBlock } from "./ItineratorBlock";

new ItineratorBlock("content", {
  attributes,
  edit: (props) => {
    const { contentTypes, loading } = useContentTypes()
    if (loading) return null
    return (
    <Edit {...props}>
        <Sidebar contentTypes={contentTypes} />
        <Display contentTypes={contentTypes} />
    </Edit>
    )
  },
});

const useContentTypes = () => {
  const [contentTypes, setContentTypes] = useState([]);
  const { itinPageType } = useItinPageMeta();
  const [loading, setLoading] = useState(true);

  useEffect(() => {
    if (!!itinPageType) {
      fetch(`/wp-json/itinerator/v1/content_types/${itinPageType}`)
        .then((res) => res.json())
        .then((contentTypes) => {
          setContentTypes(contentTypes);
        })
        .then(() => setLoading(false));
    }
  }, [itinPageType]);
  return { contentTypes, loading }
}

const useContentTypeType = (contentTypes) => {
  const { attributes } = useEditContext();
  const { contentType } = attributes;
  const { type } = contentTypes.find(t => t.name === contentType) || {}
  return type
}

const Sidebar = ({ contentTypes }) => {
  const { attributes, setAttributes } = useEditContext();
  const { showDescription, contentType, ...otherAttributes } = attributes
  const options = [
    { value: "", label: "Select Content Type" },
    ...contentTypes.map(({ name }) => ({
      value: name,
      label: startCase(name),
    })),
  ]
  return (
    <InspectorControls>
      <OptionsPanel>
        <ToggleControls custom_attributes={otherAttributes} />
        <PanelRow>
          <ToggleControl label="Show Content?" checked={showDescription} onChange={(val) => setAttributes({ showDescription: val })} />
        </PanelRow>
        <PanelRow>
          <SelectControl
            label="Content Type"
            value={contentType}
            onChange={(type) => setAttributes({ contentType: type })}
            options={options}
          />
        </PanelRow>
      </OptionsPanel>
    </InspectorControls>
  );
};

const Display = ({ contentTypes }) => {
  const contentTypeType = useContentTypeType(contentTypes)
  const { itinPageType } = useItinPageMeta();
  const { attributes } = useEditContext();
  const { showTitle, showDescription, contentType } = attributes;
  const { title, description, media } = SAMPLE_CONTENT;
  const blockProps = useBlockProps({
    className: `col-12 text-center itinerator-${itinPageType} content ${contentType}`,
  });
  return (
    <div {...blockProps}>
      {showTitle && <h3>{title}</h3>}
      {!!contentTypeType && showDescription && (
        <Fragment>
          {contentTypeType.includes("Markdown") && <p>{description}</p>}
          {contentTypeType.includes("Media") && <img src={media} />}
        </Fragment>
      )}
    </div>
  );
};