// Import necessary components and modules
import {
  InspectorControls,
  useBlockProps
} from "@wordpress/block-editor";
import {
  Edit,
  OptionsPanel,
  ToggleControls,
  useEditContext,
} from "./common";
import attributes from "../attributes/dmo-itineraries.json";
import { ItineratorBlock } from "./ItineratorBlock";
import { useItinPageMeta, useRecords } from "../hooks";
import { useItinSample } from "./store";
// import Flickity from "react-flickity-component";
import { PanelRow, SelectControl, __experimentalHeading as Heading, __experimentalGrid as Grid, __experimentalUnitControl as UnitControl } from "@wordpress/components";
import { Fragment } from "@wordpress/element";

// Create an instance of the ItineratorBlock class with attributes and support options
new ItineratorBlock("dmo-itineraries", {
  attributes,
  supports: {
    color: {
      text: false,
      background: false,
      link: false,
    },
    ariaLabel: true,
    customClassName: false,
  },
  edit: (props) => {
    // Get page type and page ID information
    const { itinPageType, itinPageId } = useItinPageMeta();
    // Fetch sample data based on page type and ID
    const { sample, loading } = useItinSample(itinPageType, itinPageId);
    if (loading) return null;

    // Render the Edit component, which includes Sidebar and Display components
    return (
      <Edit {...props}>
        <Sidebar />
        <Display sample={sample} />
      </Edit>
    );
  },
});

// Define the Sidebar component
const Sidebar = () => {
  const { attributes, setAttributes } = useEditContext();
  const { itinPageType } = useItinPageMeta();
  // Destructure attributes from the 'attributes' object
  const { viewType, columns, border, platformLogoBorderRadius, overlayBorderRadius, gridItemBorderRadius, gridGap, ...otherAttributes } = attributes;

  // Render InspectorControls including OptionsPanel, ToggleControls
  return (
    // This section contains InspectorControls, used for block settings in the block editor.
    <InspectorControls key="settings">
      <OptionsPanel>

        {/* ToggleControls component based on itinPageType condition */}
        <ToggleControls custom_attributes={itinPageType === 'dmo-itineraries' ? attributes : otherAttributes} />

        {/* PanelRow containing a SelectControl for choosing a view type (style) */}
        <PanelRow>
          <SelectControl
            label="Layout"
            value={viewType}
            onChange={(type) => setAttributes({ viewType: type })}
            options={[
              { value: "carousel", label: "Carousel" },
              { value: "grid", label: "Grid" },
            ]}
          />
        </PanelRow>
        {/* Conditional rendering based on the selected view type */}
        {viewType === "carousel" && (
          <Fragment>
            {/* Content to display when the "carousel" view type is selected */}
          </Fragment>
        )}

        {/* Can add this conditon back when the carousel is working better */}
        {viewType === "grid" && (
          <Fragment>
            {/* PanelRow containing SelectControl for choosing the number of columns */}
            <PanelRow>
              <div style={{ minWidth: "50%" }}>
                <SelectControl
                  label="Columns"
                  value={columns}
                  onChange={(number) => setAttributes({ columns: number })}
                  options={[
                    { value: "auto", label: "Auto - Display all in one row" },
                    { value: "1", label: "1" },
                    { value: "2", label: "2" },
                    { value: "3", label: "3" },
                    { value: "4", label: "4" },
                    { value: "5", label: "5" },
                    { value: "6", label: "6" },
                  ]}
                />
              </div>
            </PanelRow>

          </Fragment>
        )}
        <UnitControl
          label="Grid Gap"
          value={gridGap}
          onChange={(gap) => setAttributes({ gridGap: gap })}
        />
        <UnitControl
          label="Platform Logo Border Radius"
          value={platformLogoBorderRadius}
          onChange={(radius) => setAttributes({ platformLogoBorderRadius: radius })}
        />
        <UnitControl
          label="Overlay Border Radius"
          value={overlayBorderRadius}
          onChange={(radius) => setAttributes({ overlayBorderRadius: radius })}
        />
        <UnitControl
          label="Grid Item Border Radius"
          value={gridItemBorderRadius}
          onChange={(radius) => setAttributes({ gridItemBorderRadius: radius })}
        />
      </OptionsPanel>
    </InspectorControls>
  );

};

// Define the Display component to render the dmos
const Display = ({ sample }) => {
  const { attributes } = useEditContext();
  const { itinPageId, itinPageType } = useItinPageMeta();
  const id = itinPageId || (sample && sample.id.toString());
  const { dmoPartnerApiKey } = itineratorGlobal;
  const { viewType, columns, border, platformLogoBorderRadius, overlayBorderRadius, gridItemBorderRadius, gridGap, ...otherAttributes } = attributes;

  // Fetch records based on page type and ID
  const { records, loading } = useRecords(
    `dmos/itineraries`,
    {
      secret_key: dmoPartnerApiKey,
    },
    id
  );

  // Define blockProps to apply to the surrounding container
  const blockProps = useBlockProps({
    className: "itinerator-dmos alignfull itinerator-dmo-carousel",
  });

  // Filter records based on page type and render the dmos
  const filtered_records = records.filter(record => record.status === 'active');

  return (
    // This section contains the main content structure of the component.
    <div {...blockProps}>

      {/* Container for organizing itineraries in columns. */}
      <div className="wp-block-columns">
        {/* Display a message if there are no itineraries and no loading is in progress. */}
        {!filtered_records.length && !loading && "No itineraries found"}

        {/* Map and display itineraries in columns. */}
        <Grid columns={(attributes.columns === 'auto') ? 4 : attributes.columns} columnGap={attributes.gridGap}>

          {filtered_records.map((itinerary) => (
            <div className="item" key={itinerary.id}  >
              <figure
                style={itinerary.hero && { backgroundImage: `url(${itinerary.hero.url})`, padding: '2rem', backgroundSize: 'cover', margin: '0', height: '100%', borderRadius: gridItemBorderRadius }}
              >

                <div className="platform-logo">
                  <figure
                    style={{ textAlign: "center" }}
                  >
                    {itinerary.platform_logo && (
                      <img src={itinerary.platform_logo.url} alt="" style={itinerary.platform_logo.url && { padding: '2rem', backgroundSize: 'contain', margin: '0', height: '100px', backgroundRepeat: 'no-repeat', backgroundPosition: 'center', background: 'rgba(255, 255, 255, 0.7)', borderRadius: platformLogoBorderRadius }} />
                    )}

                  </figure>
                </div>
                <div className="overlay-content" style={{
                  background: 'rgba(26,32,44,0.5)',
                  margin: '10px', color: '#fff', textShadow: '1px 1px 3px rgba(0,0,0,0.5)', borderRadius: overlayBorderRadius,
                }}>

                  <div className="itinerary-details" style={{ padding: '1rem', borderRadius: '10px' }}>

                    {itinerary.subtitle &&
                      <h2 style={{color: "#ffffff" }}>
                        {itinerary.subtitle == '' || itinerary.subtitle == "Itinerary"
                          ? "Discover more about this itinerary."
                          : itinerary.subtitle
                        }
                      </h2>
                    }

                    {itinerary.stop_count && (
                      <p>
                        {itinerary.stop_count} Stops, {itinerary.leg_count} {itinerary.leg_count > 1 ? 'Days' : 'Day'}
                      </p>
                    )}
                    {itinerary.main_content && <p>{itinerary.main_content}</p>}
                  </div>
                </div>
              </figure>
            </div>
          ))}


        </Grid>
      </div>
    </div>
  );

};
