// Import necessary components and modules
import {
  InspectorControls,
  useBlockProps
} from "@wordpress/block-editor";
import {
  Edit,
  OptionsPanel,
  ToggleControls,
  useEditContext,
} from "./common";
import attributes from "../attributes/dmos.json";
import { ItineratorBlock } from "./ItineratorBlock";
import { useItinPageMeta, useRecords } from "../hooks";
import { useItinSample } from "./store";
import Flickity from "react-flickity-component";
import { flickity_options } from "../constants";
import { PanelRow, SelectControl } from "@wordpress/components";
import { Fragment } from "@wordpress/element";

// Create an instance of the ItineratorBlock class with attributes and support options
new ItineratorBlock("dmos", {
  attributes,
  supports: {
    color: {
      text: false,
      background: false,
      link: true,
    },
    align: true,
    ariaLabel: true,
    customClassName: true,
  },
  edit: (props) => {
    // Get page type and page ID information
    const { itinPageType, itinPageId } = useItinPageMeta();
    // Fetch sample data based on page type and ID
    const { sample, loading } = useItinSample(itinPageType, itinPageId);
    if (loading) return null;

    // Render the Edit component, which includes Sidebar and Display components
    return (
      <Edit {...props}>
        <Sidebar />
        <Display sample={sample} />
      </Edit>
    );
  },
});

// Define the Sidebar component
const Sidebar = () => {
  const { attributes, setAttributes } = useEditContext();
  const { itinPageType } = useItinPageMeta();
  // Destructure attributes from the 'attributes' object
  const { viewType, columns, border, ...otherAttributes } = attributes;

  // Render InspectorControls including OptionsPanel, ToggleControls
  return (
    // This section contains InspectorControls, used for block settings in the block editor.
    <InspectorControls key="settings">
      <OptionsPanel>
        {/* ToggleControls component based on itinPageType condition */}
        <ToggleControls custom_attributes={itinPageType === 'dmo' ? attributes : otherAttributes} />

        {/* PanelRow containing a SelectControl for choosing a view type (style) */}
        <PanelRow>
          <SelectControl
            label="Layout"
            value={viewType}
            onChange={(type) => setAttributes({ viewType: type })}
            options={[
              { value: "carousel", label: "Carousel" },
              { value: "grid", label: "Grid" },
            ]}
          />
        </PanelRow>

        {/* Conditional rendering based on the selected view type */}
        {viewType === "carousel" && (
          <Fragment>
            {/* Content to display when the "carousel" view type is selected */}
          </Fragment>
        )}

        {viewType === "grid" && (
          <Fragment>
            {/* PanelRow containing SelectControl for choosing the number of columns */}
            <PanelRow>
              <div style={{ minWidth: "50%" }}>
                <SelectControl
                  label="Columns"
                  value={columns}
                  onChange={(type) => setAttributes({ columns: type })}
                  options={[
                    { value: "1", label: "1" },
                    { value: "2", label: "2" },
                    { value: "3", label: "3" },
                    { value: "4", label: "4" },
                  ]}
                />
              </div>
            </PanelRow>
          </Fragment>
        )}
      </OptionsPanel>
    </InspectorControls>
  );

};

// Define the Display component to render the dmos
const Display = ({ sample }) => {
  const { attributes, setAttributes } = useEditContext();
  const { showName, showImage, viewType } = attributes;
  const { itinPageId, itinPageType } = useItinPageMeta();
  const id = itinPageId || (sample && sample.id.toString());
  const columnWidth = (100 / (attributes.columns === 'auto') ? 4 : attributes.columns);

  // Fetch records based on page type and ID
  const { records, loading } = useRecords(
    `dmos`,
    {
      region_id: itinPageType === 'region' ? id : null,
    },
    id
  );

  // Define blockProps to apply to the surrounding container
  const blockProps = useBlockProps({
    className: `itin itinerator-dmos ${attributes.align ? `align${attributes.align}` : ''} ${attributes.customClassName ? `${attributes.customClassName}` : ''}`,
  });

  // Filter records based on page type and render the dmos
  const filtered_records = itinPageType === 'dmo' ? records.filter(r => r.id !== parseInt(id)) : records;



  return (
    // This section contains the main content structure of the component.
    <div {...blockProps}>
      {/* Container for organizing DMOs in columns. */}
      <div className="container-fluid">
        {/* Display a message if there are no DMOs and no loading is in progress. */}
        {!filtered_records.length && !loading && "No DMOs found"}

        {/* Map and display DMOs in columns. */}
        {viewType === "grid" && (
          <div className="row g-4">
            {filtered_records.map((dmo) => (
              <DmoCard key={dmo.id} dmo={dmo} />
            ))}
          </div>
        )}
        {viewType === "carousel" && (
          <div>
            <Flickity options={flickity_options} data-flickity='{ "wrapAround": true }'>
              {filtered_records.map((dmo) => (
                <DmoCard key={dmo.id} dmo={dmo} />
              ))}
            </Flickity>
          </div>
        )}
      </div>
    </div>
  );
};

const DmoCard = ({ dmo }) => {
  const { attributes, setAttributes } = useEditContext();
  const { showName, showImage, viewType } = attributes;
  function getColumnClasses(columns) {
    switch (columns) {
      case "1":
        return 'col-12';
      case "2":
        return 'col-md-6';
      case "3":
        return 'col-md-4';
      case "4":
        return 'col-md-3';
      default:
        return 'col-md-3';
    }
  }
  return (
    <div
                key={dmo.id}
                // Apply CSS classes
                className={`col-12 ${getColumnClasses(attributes.columns)} mb-4`}
              >
                <div className="card h-100 p-0">
                  {dmo.logo && showImage && (
                    <div className="card-img-top dmos-image-container text-center">
                      <img className="img-fluid dmos-image" src={dmo.logo.url} alt="" />
                    </div>
                  )}
                  {!dmo.logo && showImage && (
                    <div className="d-flex align-items-center justify-content-center img-fluid rounded-top  dmos-image dmos-image-empty">
                      <span className="text-muted">No image available</span>
                    </div>
                  )}
                  {dmo.name && showName && (
                    <div className="card-body dmos-content">
                      <h4 className="card-title dmo-name text-center ">
                        <a href={dmo.website}>{dmo.name}</a>
                      </h4>
                    </div>
                  )}
                  {/* {dmo.description && (
                    <p className="card-text dmo-description">{dmo.description}</p>
                  )} */}
                </div>
              </div>
  );
};
