import { InspectorControls, useBlockProps, RichText } from "@wordpress/block-editor";
import {
	Edit,
	LinkedPageSelect,
	OptionsPanel,
	TagFilters,
	ToggleControls,
	useEditContext,
} from "./common";
import attributes from "../attributes/events.json";
import { ItineratorBlock } from "./ItineratorBlock";
import { useItinPageMeta, useRecords } from "../hooks";
import { TagCategoryControl } from "./common/TagCategoryControl";
import { useItinSample } from "./store";
import { formatEventTimeDates } from "../helpers";

new ItineratorBlock("events", {
	attributes,
  supports: {
    align: true,
    customClassName: true,
  },
	edit: (props) => {
		const { itinPageType, itinPageId } = useItinPageMeta();
		const { sample, loading } = useItinSample(itinPageType, itinPageId);
		if (loading) return null;
		return (
			<Edit {...props}>
				<Sidebar />
				<Display sample={sample} />
			</Edit>
		);
	},
});

const Sidebar = () => {
	return (
		<InspectorControls key="settings">
			<OptionsPanel>
				<LinkedPageSelect type="event" />
				<ToggleControls />
				<TagCategoryControl />
			</OptionsPanel>
			<TagFilters />
		</InspectorControls>
	);
};

const Display = ({ sample }) => {
	const { attributes, setAttributes } = useEditContext();
	const {
		matchRecordsByTagCategory,
		showImage,
		showName,
		showDescription,
		buttonText,
		showDateFilter
	} = attributes;
	const { itinPageId, itinPageType } = useItinPageMeta();
	const id = itinPageId || (sample && sample.id.toString());

	const blockProps = useBlockProps({
		className: `itin itinerator-events ${attributes.align ? `align${attributes.align}` : ''} ${attributes.customClassName ? `${attributes.customClassName}` : ''}`,
	});
	const { records, loading } = useRecords(
		`events`,
		{ region_id: itinPageType === 'region' && !matchRecordsByTagCategory ? id : null },
		id
	);
	return (
    <div {...blockProps}>
        <div className="container-fluid">
            {showDateFilter && (
                <div class="filters mb-4">
                    <div class="date-filter-wrapper" style="display: flex; flex-direction: row; align-items: stretch; gap: 0.5rem;">
                        <input type="text" id="dateFilter" class="form-control" placeholder="Select dates..."/>
                        <button type="button" class="btn btn-secondary clear-dates">Clear Dates</button>
                        <div class="date-preset-ranges mb-3" style="display: contents;">
                      <button type="button" class="btn btn-secondary" data-range="today">Today</button>
                      <button type="button" class="btn btn-secondary" data-range="week">Next 7 Days</button>
                      <button type="button" class="btn btn-secondary" data-range="month">Next Month</button>
                      <button type="button" class="btn btn-secondary" data-range="3months">Next 3 Months</button>
                    </div>
                </div>
              </div>
            )}
            {!records.length && !loading && "No Events found"}
            {records.map((event) => (
                <div
                    key={event.id}
                    className={`event-row row justify-content-end align-items-center searchable-item mb-4 ${event.slug} ${event.region_slug} ${event.tags.map((t) => `tag-id-${t.tag_id}`).join(" ")}`}
                >
                    <div className="col-12 col-md-6 order-md-2 p-0 open-half">
                        {event.hero && (
                            <div className="image">
                                <img src={event.hero.url} className="img-fluid" />
                            </div>
                        )}

                    </div>
                    <div className="col-12 col-md-6 order-md-1 contained-half">
                        <div className="content">
                            {event.name && <h2>{event.name}</h2>}
                            {event.subtitle && attributes.showSubtitle && <h3>{event.subtitle}</h3>}
                            {event.short_description && attributes.showDescription && <p>{event.short_description}</p>}
                            {!event.short_description && attributes.showDescription && event.description && <div dangerouslySetInnerHTML={{ __html: event.description }} />}
                            {event.name && (
                                <a
                                    // href={`/event/${event.slug}`}
                                    className="btn btn-primary"
                                    aria-label={`Explore ${event.name}`}
                                >
                                    <RichText
                                        tagName="span"
                                        value={buttonText}
                                        onChange={(t) => setAttributes({ buttonText: t })}
                                    />{" "}
                                    {event.name}
                                </a>
                            )}
                        </div>
                    </div>
                </div>
            ))}
        </div>
    </div>
);
}


