import { useState, useEffect } from "react";
import { __ } from "@wordpress/i18n";
import { startCase } from "lodash";
import { TextControl, SelectControl, PanelRow } from "@wordpress/components";
import { InspectorControls, useBlockProps } from "@wordpress/block-editor";

import { ItineratorBlock } from "./ItineratorBlock";
import attributes from "../attributes/filter.json";
import { listToSelectOptions } from "../helpers";
import {
  useTagCategories,
  useAfterMountEffect,
  useRecordListOptions,
  useTagsByCategoryOptions,
} from "../hooks";
import { Edit, OptionsPanel, useEditContext } from "./common";

new ItineratorBlock("filter", {
  attributes,
  supports: {
    align: false,
    customClassName: true,
  },
  edit: (props) => (
    <Edit {...props}>
      <Sidebar />
      <Display />
    </Edit>
  ),
});

const Sidebar = () => {
  const { attributes, setAttributes } = useEditContext();
  const { filterType, filterPlaceholder, tagCategory } = attributes;
  const setTagCategory = (tagCategory) => setAttributes({ tagCategory });
  const { defaultCategory, options } = useTagCategories();
  const needsCategory =
    filterType === "tag-category" &&
    !!defaultCategory &&
    !!options.length &&
    options.filter(({ value }) => value === tagCategory).length === 0;
  useEffect(() => {
    if (filterType !== "tag-category" || needsCategory) {
      setTagCategory(needsCategory ? defaultCategory : "");
    }
  }, [filterType, needsCategory, defaultCategory]);

  useAfterMountEffect(() => {
    let placeholder;
    switch (filterType) {
      case "search":
        placeholder = "Search";
        break;
      case "region":
        placeholder = "Select Region";
      case "tag-category":
        placeholder = "Select Tag";
      default:
        placeholder = `Select ${startCase(filterType)} `;
        break;
    }
    setAttributes({ filterPlaceholder: placeholder });
  }, [filterType]);

  return (
    <InspectorControls key="settings">
      <OptionsPanel>
        <PanelRow>
          <SelectControl
            label="Filter Type"
            value={filterType}
            onChange={(type) => setAttributes({ filterType: type })}
            options={listToSelectOptions(["search", "region", "tag-category"])}
            id="itinFilterType"
          />
        </PanelRow>
        <PanelRow>
          <TextControl
            label="Filter Placeholder"
            value={filterPlaceholder}
            onChange={(filterPlaceholder) =>
              setAttributes({ filterPlaceholder })
            }
          />
        </PanelRow>
        {filterType === "tag-category" && (
          <PanelRow>
            <SelectControl
              label="Tag Category"
              value={tagCategory}
              onChange={setTagCategory}
              options={options}
              id="itinTagCategory"
            />
          </PanelRow>
        )}
      </OptionsPanel>
    </InspectorControls>
  );
};

const Display = () => {
  const { attributes } = useEditContext();
  const { filterType, filterPlaceholder, tagCategory } = attributes;
  const [search, setSearch] = useState("");
  const [option, setOption] = useState("");
  const regionOptions = useRecordListOptions("regions", filterPlaceholder);
  const { tagOptions } = useTagsByCategoryOptions(
    tagCategory,
    filterPlaceholder
  );

  const options = {
    search: [],
    region: regionOptions,
    "tag-category": tagOptions,
  };
  const blockProps = useBlockProps({
    className: "container itinerator itinerator-filter",
  });

  return (
    <div {...blockProps}>
      {filterType === "search" ? (
        <TextControl
          value={search}
          onChange={setSearch}
          placeholder={filterPlaceholder}
          id="itinSearch"
        />
      ) : (
        <SelectControl
          value={option}
          onChange={setOption}
          options={options[filterType]}
          id={`itinSelect-${tagCategory || filterType}`}
          data-itin-filter="true"
          data-itin-filter-type={filterType}
        />
      )}
    </div>
  );
};
