import { __ } from "@wordpress/i18n";
import { InspectorControls, useBlockProps } from "@wordpress/block-editor";
import { SelectControl, PanelBody } from "@wordpress/components";
import { useEffect} from "@wordpress/element";
import { Edit, OptionsPanel, useEditContext, ToggleControls } from "./common";
import attributes from "../attributes/form-block.json";
import { ItineratorBlock } from "./ItineratorBlock";
import { useItinPageMeta, useRecords } from "../hooks";
import { useItinSample } from "./store";

new ItineratorBlock("form-block", {
  attributes,
  supports: {
    align: true,
    customClassName: true,
  },
  edit: (props) => (
    <Edit {...props}>
      <Sidebar />
      <Display />
    </Edit>
  ),
});

const Sidebar = () => {
  const { itinPageId, itinPageType } = useItinPageMeta();
  const { records, loading } = useRecords(
    `forms`,
    {
      region_id:
        itinPageType === "region" && !matchRecordsByTagCategory ? id : null,
    },
    id
  );

  const { attributes, setAttributes } = useEditContext();
  const { selectedForm } = attributes;

  const formOptions = records.map((form) => ({
    label: form.name,
    value: form.id.toString(),
  }));

  // Check if there is only one form in the records array
  useEffect(() => {
    if (records.length === 1) {
      setAttributes({ selectedForm: records[0].id.toString() });
    } else if (records.length > 1 && !selectedForm) {
      setAttributes({ selectedForm: records[0].id.toString() });
    }
  }, [records, setAttributes, selectedForm]);

  return (
    <InspectorControls>
      <PanelBody title={__("Form Settings", "itinerator-plugin")}>
        {!loading && (
          <SelectControl
            label={__("Select Form", "itinerator-plugin")}
            value={selectedForm}
            options={formOptions}
            onChange={(value) => setAttributes({ selectedForm: value })}
          />
        )}
      </PanelBody>
      <OptionsPanel>
        <ToggleControls />
      </OptionsPanel>
    </InspectorControls>
  );
};



const Display = () => {
  const { itinPageType } = useItinPageMeta();
  const { sample, loading } = useItinSample(itinPageType);
  const { attributes } = useEditContext();
  const { showName, showDescription, showLocation, showDates, selectedForm } = attributes;

  const { records } = useRecords(
    `forms`,
    {
      region_id:
        itinPageType === "region" && !matchRecordsByTagCategory ? id : null,
    },
    id
  );

  const blockProps = useBlockProps({
    className: `itin itinerator-form ${attributes.align ? `align${attributes.align}` : ''} ${attributes.customClassName ? `${attributes.customClassName}` : ''}`,
  });

  if (loading) return null;

  const selectedFormData = records.find((form) => form.id === parseInt(selectedForm));


  if (!selectedForm) {
    return (
      <div {...blockProps}>
        <div className="container-fluid">
          <div className="alert alert-info">
            {__('Please select a form in the block settings.', 'my-plugin')}
          </div>
        </div>
      </div>
    );
  }

  if (!selectedFormData) return null;

  return (
    <div {...blockProps}>
      <div className="container-fluid">
        {selectedFormData.form_fields.map((field) => (
          <div key={field.id} className="form-group">
            <label htmlFor={`field-${field.id}`}>{field.label}</label>
            {field.type === 'TextField' && (
              <input type="text" className="form-control" id={`field-${field.id}`} disabled />
            )}
            {field.type === 'NumberField' && (
              <input type="number" className="form-control" id={`field-${field.id}`} disabled />
            )}
            {field.type === 'DateTimeField' && (
              <input type="datetime-local" className="form-control" id={`field-${field.id}`} disabled />
            )}
            {field.type === 'DateField' && (
              <input type="date" className="form-control" id={`field-${field.id}`} disabled />
            )}
            {field.type === 'ImageField' && (
              <div className="custom-file">
                <input type="file" className="custom-file-input" id={`field-${field.id}`} disabled />
                <label className="custom-file-label" htmlFor={`field-${field.id}`}>Choose file</label>
              </div>
            )}
            {field.type === 'SelectField' && (
              <select className="form-control" id={`field-${field.id}`} disabled>
                <option value="">Select an option</option>
                {field.options.choices.map((choice, index) => (
                  <option key={index} value={choice}>
                    {choice}
                  </option>
                ))}
              </select>
            )}
            {field.type === 'BooleanField' && (
              <div className="form-check">
                <input type="checkbox" className="form-check-input" id={`field-${field.id}`} disabled />
                <label className="form-check-label" htmlFor={`field-${field.id}`}>
                  {field.label}
                </label>
              </div>
            )}
            {field.description && <p className="form-text">{field.description}</p>}
          </div>
        ))}
      </div>
    </div>
  );
};




