import { __ } from "@wordpress/i18n";
import { InspectorControls, useBlockProps, RichText } from "@wordpress/block-editor";
import { Edit, LinkedPageSelect, OptionsPanel, TagFilters, useEditContext } from "./common";
import attributes from "../attributes/itineraries.json";
import { ItineratorBlock } from "./ItineratorBlock";
import { useItinPageMeta, useRecords } from "../hooks";
import pluralize from "pluralize";
import { useItinSample } from "./store";
import { TagCategoryControl } from "./common/TagCategoryControl";
import { FilterRecordsRegion } from "./common/FilterRecordsRegion";

new ItineratorBlock("itineraries", {
  attributes,
  supports: {
    align: true,
    customClassName: true,
  },
  edit: (props) => {
    const { itinPageType, itinPageId } = useItinPageMeta();
    const { sample, loading } = useItinSample(itinPageType, itinPageId);
    if (loading) return null;
    return (
      <Edit {...props}>
        <Sidebar />
        <Display sample={sample} />
      </Edit>
    );
  },
});

const Sidebar = () => {
  return (
    <InspectorControls key="settings">
      <OptionsPanel>
        <LinkedPageSelect type="itinerary" />
        <FilterRecordsRegion />
        <TagCategoryControl />
      </OptionsPanel>
      <TagFilters />
    </InspectorControls>
  );
};

const Display = ({ sample }) => {
  const { itinPageId, itinPageType } = useItinPageMeta();
  const { attributes, setAttributes } = useEditContext();
  const { matchRecordsByTagCategory, buttonText } = attributes;
  const id = itinPageId || (sample && sample.id.toString());
  const { records } = useRecords(
    `itineraries`,
    {
      region_id:
        itinPageType === "region" && !matchRecordsByTagCategory ? id : null,
    },
    id
  );
  const blockProps = useBlockProps({
    className: `itin itinerator-itineraries ${attributes.align ? `align${attributes.align}` : ''} ${attributes.customClassName ? `${attributes.customClassName}` : ''}`,
  });
  return (
    <div {...blockProps}>
      <div className="container-fluid">
        {records.map((record) => (
          <div
            key={record.id}
            className={`itinerary-item row justify-content-end align-items-center searchable-item mb-4 ${record.slug}`}
          >
            <div className="col-12 col-md-6 order-md-2 p-0">
              {record.hero && (
                <div className="image itinerary-hero-container">
                  <img src={record.hero.url} className="img-fluid  itinerary-hero-image" />
                </div>
              )}
              <a href={record.path} className="stretched-link"></a>
            </div>
            <div className="col-12 col-md-6 order-md-1">
              <div className="content itinerary-content">
                <h3 className="itinerary-name">{record.name}</h3>
                <div>
                  <span className="itinerary-counts">
                    {`${pluralize("Stop", record.stop_count, true)} | ${pluralize(
                      "Day",
                      record.leg_count,
                      true
                    )}`}
                  </span>
                  <span className="itinerary-region-name">
                    {record.region_name}
                  </span>
                </div>
                <p className="itinerary-description">{record.description}</p>
                <a
                  href="#"
                  className="btn btn-primary itinerary-button"
                  aria-label={`Explore ${record.name}`}
                >
                  <RichText
                    tagName="span"
                    value={buttonText}
                    onChange={(t) => setAttributes({ buttonText: t })}
                  /> {record.name}
                </a>
              </div>
            </div>
          </div>
        ))}
      </div>
    </div>
  );
};
