import { __ } from "@wordpress/i18n";
import { useBlockProps } from "@wordpress/block-editor";
import { SelectControl } from "@wordpress/components";
import { startCase } from "lodash";
import { initialArray } from "../helpers";
import { useItinOptions } from "./store";
import { useItinPageMeta, usePostType } from "../hooks";
// import { ServerSideRender } from "@wordpress/editor";
import { InspectorControls } from "@wordpress/block-editor";
import { Edit, LinkedPageSelect, OptionsPanel, useEditContext } from "./common";
import attributes from "../attributes/itinerary.json";
import { ItineratorBlock } from "./ItineratorBlock";
import { useDispatch } from "@wordpress/data";
import React, { useEffect } from 'react';
import { useItinSample } from "./store";


const { pluginDirUrl } = itineratorGlobal;

new ItineratorBlock("itinerary", {
  attributes,
  supports: {
    align: true,
    customClassName: true,
  },
  edit: (props) => (
    <Edit {...props}>
      <Sidebar />
      <Display />
    </Edit>
  ),
});

// Define Sidebar as a functional component
const Sidebar = () => {
  // Destructure and obtain metadata for the itinPage from a custom hook
  const { itinPageType: rawItinPageType, itinPageId } = useItinPageMeta();

  // Fallback to 'itinerary' so that we can use the specificItinerary dropdown in block sidebar
  const itinPageType = rawItinPageType || 'itinerary';

  // Get the editPost function from redux
  const { editPost } = useDispatch("core/editor");

  // Get options and loading state for the current itin page
  const { options, loading } = useItinOptions(itinPageType);

  // Prepare the select box options by combining initial array with the options object
  const select_options = initialArray("None").concat(
    options.map(({ name, slug, id }) => ({ value: id, label: name }))
  );

  // Get the attributes and the function to set them from the editing context
  const { attributes, setAttributes } = useEditContext();

  // Seperate 'specificItinerary' and other attributes from the attributes object
  const { specificItinerary, ...otherAttributes } = attributes;

  // monitor rawItinPageType and set specificItinerary to "" when the page type is used.
  useEffect(() => {
    if (rawItinPageType) {
      setAttributes({ specificItinerary: "" });
    }
  }, [rawItinPageType]);

  {
    !rawItinPageType &&
      <SelectControl
        label={`Specific ${startCase('itinerary')}`}
        value={attributes.specificItinerary}
        onChange={(itin) => setAttributes({ specificItinerary: itin })}
        options={select_options}
      />
  }

  return (
    <InspectorControls key="settings">
      <OptionsPanel>
        {
          !rawItinPageType &&
          <SelectControl
            label={`Specific ${startCase('itinerary')}`}
            value={attributes.specificItinerary}
            onChange={(itin) => setAttributes({ specificItinerary: itin })}
            options={select_options}
            />
          }
      </OptionsPanel>
    </InspectorControls>
  );
};


const Display = () => {
  const { itinPageType: rawItinPageType } = useItinPageMeta();
  const itinPageType = rawItinPageType || 'itinerary';
  const { sample, loading } = useItinSample(itinPageType);

  const blockProps = useBlockProps({
    className: `itin itinerator-itinerary ${attributes.align ? `align${attributes.align}` : ''} ${attributes.customClassName ? `${attributes.customClassName}` : ''}`,
  });

  if (loading) return null;

  return (
    <div {...blockProps}>
      Note: Sample data displayed in editor: real itineary will display on front end.
      <section className="section-itinerary anchor-point" id="trail">
        <div className="container-fluid">
          <div className="tab-content row">
            <div
              className="tab-pane active d-md-block col-12 col-md-6"
              id="itinerary"
              role="tabpanel"
              aria-labelledby="itinerary-tab"
            >
              <div className="row m-0">
                <div className="col-12 itinerary-content">
                  <div className="group-start mt-3">
                    <div className="media">
                      <div className="media-body">
                        <h2 className="itineray-name">{sample.name}</h2>
                        <div
                          className="has-md-font-size"
                          dangerouslySetInnerHTML={{ __html: sample.description }}
                        />
                      </div>
                    </div>
                  </div>
                  <div>
                    {sample.legs && sample.legs.map((leg, legIndex) => (
                      <React.Fragment key={`title` + legIndex}>
                        <div className="leg-title">{leg.title}</div>
                        <ul className="list-group list-group-flush">
                          {leg.stops && leg.stops.map((stop, stopIndex) => (
                          <React.Fragment key={`leg-` + stopIndex + legIndex}>
                            {stop.main_type === 'stop' && (
                              <li
                                key={`stop` + stopIndex}
                                className="list-group-item"
                                data-poi-index={stop.total_stop_number}
                              >
                                <div className="badge-col">
                                  <div className="badge badge-pill">{stop.total_stop_number}</div>
                                </div>
                                <div className="content">
                                  <div className="h4">
                                    <a className="title-link">{stop.name}</a>
                                  </div>
                                  {stop.target_type === 'Listing' && stop.listing && (
                                    <div className="contact-details">
                                      {stop.listing.full_address && (
                                        <div className="address">{stop.listing.full_address}</div>
                                      )}
                                      {stop.listing.website && (
                                        <a
                                          className="t-book-now-itinerary"
                                          href={stop.listing.website}
                                          target="_blank"
                                          rel="noreferrer"
                                        >
                                          {stop.listing.website}
                                        </a>
                                      )}
                                    </div>
                                  )}
                                  <div
                                    className="description"
                                    dangerouslySetInnerHTML={{ __html: stop.description }}
                                  />
                                </div>
                              </li>
                            )}
                            {stop.main_type === 'note' && (
                              <div
                                key={`note-${stopIndex}`}
                                className="media-section"
                              >
                                <div className="media-body">
                                  {stop.hero && (
                                    <img src={stop.hero.url} className="w-100 mb-3" />
                                  )}
                                  <div
                                    className="note-description"
                                    dangerouslySetInnerHTML={{ __html: stop.description }}
                                  />
                                </div>
                              </div>
                            )}
                            </React.Fragment>
                          ))}
                        </ul>
                      </React.Fragment>
                    ))}
                  </div>
                </div>
              </div>
            </div>
            <div className="tab-pane d-md-block col-12 col-md-6" id="map" role="tabpanel" aria-labelledby="map-tab">
              <div className="map-container sticky-top">
                <div className="google-map">
                  <div className="map-internals" style={{ display: 'none' }}>
                    {sample.legs && sample.legs.flatMap((leg, legIndex) =>
                      leg.stops && leg.stops.map((stop, stopIndex) => (
                        <div
                          key={`map-${legIndex}-${stopIndex}`}
                          className="marker"
                          data-lat={stop.listing && stop.listing.lat}
                          data-lng={stop.listing && stop.listing.lng}
                          data-title={stop.name}
                          data-poi-index={stop.total_stop_number}
                          data-part={leg.order_on_itin}
                        >
                          <div data-marker>
                            <span className="badge map-badge badge-pill">{stop.total_stop_number}</span>
                          </div>
                          <div data-infowindow>
                            <div className="media media-infowindow">
                              <div className="media-body">
                                <div className="title">
                                  <a className="title-link" >
                                    {stop.name}
                                  </a>
                                </div>
                                {stop.target_type === 'Listing' && stop.listing && (
                                  <div className="contact-details">
                                    {stop.listing.full_address && (
                                      <div className="address">{stop.listing.full_address}</div>
                                    )}
                                    {stop.listing.website && (
                                      <a
                                        className="t-book-now-itinerary"
                                        href={stop.listing.website}
                                        target="_blank"
                                        rel="noreferrer"
                                      >
                                        {stop.listing.website}
                                      </a>
                                    )}
                                  </div>
                                )}
                              </div>
                            </div>
                          </div>
                        </div>
                      ))
                    )}
                  </div>
                </div>
              </div>
            </div>
          </div>
        </div>
      </section>
    </div>
  );
};
