import classnames from "classnames";

import {
  InspectorControls,
  useBlockProps,
  useInnerBlocksProps,
  store as blockEditorStore,
} from "@wordpress/block-editor";
import { useSelect } from "@wordpress/data";
import { Edit, OptionsPanel, ToggleControls, useEditContext } from "./common";
import attributes from "../attributes/listing-buttons.json";
import supports from "../supports/listing-buttons.json";
import { ItineratorBlock } from "./ItineratorBlock";

const CHILD_BLOCK_NAME = "itinerator/listing-website-button";

const ALLOWED_BLOCKS = [CHILD_BLOCK_NAME];

const DEFAULT_BLOCK = {
  name: CHILD_BLOCK_NAME,
  attributesToCopy: [
    "backgroundColor",
    "border",
    "className",
    "fontFamily",
    "fontSize",
    "style",
    "textColor",
    "width",
  ],
};

const getClassNames = ({ fontSize, style }) => {
	return classnames('wp-block-buttons', {
		"has-custom-font-size": fontSize || (style && style.typography && style.typography.fontSize),
	})
};

new ItineratorBlock("listing-buttons", {
  attributes,
  edit: (props) => {
    return (
      <Edit {...props}>
        <Sidebar />
        <Display />
      </Edit>
    );
  },
  save: ({ attributes }) => {
    const blockProps = useBlockProps.save({
      className: getClassNames(attributes),
    });
    const innerBlocksProps = useInnerBlocksProps.save(blockProps);
    return <div {...innerBlocksProps} />;
  },
  supports,
  editorStyle: "wp-block-buttons-editor",
  style: "wp-block-buttons",
});

const Sidebar = () => {
  return (
    <InspectorControls>
      <OptionsPanel>
			<ToggleControls />
      </OptionsPanel>
    </InspectorControls>
  );
};

const Display = () => {
  const { attributes } = useEditContext();
  const { layout = {} } = attributes;
  const blockProps = useBlockProps({
    className: getClassNames(attributes),
  });
  const preferredStyle = useSelect((select) => {
    const preferredStyleVariations =
      select(blockEditorStore).getSettings()
        .__experimentalPreferredStyleVariations;
    return (
      preferredStyleVariations &&
      preferredStyleVariations.value &&
      preferredStyleVariations.value[CHILD_BLOCK_NAME]
    );
  }, []);

  const innerBlocksProps = useInnerBlocksProps(blockProps, {
    allowedBlocks: ALLOWED_BLOCKS,
    __experimentalDefaultBlock: DEFAULT_BLOCK,
    __experimentalDirectInsert: true,
    template: [
      [
        CHILD_BLOCK_NAME,
        { className: preferredStyle && `is-style-${preferredStyle}` },
      ],
    ],
    __experimentalLayout: layout,
    templateInsertUpdatesSelection: true,
  });
  return <div {...innerBlocksProps} />;
};
