import classnames from "classnames";
import { __ } from "@wordpress/i18n";
import {
  useEffect,
} from "@wordpress/element";
import {
  Button,
  ButtonGroup,
  PanelBody,
  PanelRow,
  SelectControl,
} from "@wordpress/components";
import {
  InspectorControls,
  RichText,
  useBlockProps,
  __experimentalUseBorderProps as useBorderProps,
  __experimentalUseColorProps as useColorProps,
  __experimentalGetSpacingClassesAndStyles as useSpacingProps,
} from "@wordpress/block-editor";
import { Edit, OptionsPanel, useEditContext } from "./common";
import { useItinPageMeta } from "../hooks";
import attributes from "../attributes/listing-website-button.json";
import supports from "../supports/listing-website-button.json";
import { ItineratorBlock } from "./ItineratorBlock";

const camelToHTMLClass = (str) =>
  str.replace(/[A-Z]/g, (letter) => `-${letter.toLowerCase()}`);

const stylesToString = (styles) => {
  return Object.keys(styles)
    .map((key) => `${camelToHTMLClass(key)}:${styles[key]};`)
    .join("");
};

new ItineratorBlock("listing-website-button", {
  attributes,
  edit: (props) => {
    return (
      <Edit {...props}>
        <Sidebar />
        <Display />
      </Edit>
    );
  },
  parent: ["itinerator/listing-buttons"],
  supports,
  styles: [
    { name: "fill", label: "Fill", isDefault: true },
    { name: "outline", label: "Outline" },
  ],
  editorStyle: "wp-block-button-editor",
  style: "wp-block-button",
});

const Sidebar = () => {
  const {
    attributes: { type },
    setAttributes,
  } = useEditContext();
  return (
    <InspectorControls>
      <WidthPanel />
      <OptionsPanel>
        <PanelRow>
          <SelectControl
            label="Button Type"
            value={type}
            onChange={(type) => setAttributes({ type })}
            options={[
              { value: "website", label: "Website" },
              { value: "booking_website", label: "Booking Website" },
            ]}
          />
        </PanelRow>
      </OptionsPanel>
    </InspectorControls>
  );
};

const Display = () => {
  const { attributes, setAttributes, onReplace, mergeBlocks } =
    useEditContext();
  const { width, text, style, className } = attributes;
  const { itinPageType } = useItinPageMeta();
  const blockProps = useBlockProps();
  const borderProps = useBorderProps(attributes);
  const colorProps = useColorProps(attributes);
  const spacingProps = useSpacingProps(attributes);

  const wrapperClassNames = classnames(
    `itinerator-${itinPageType}`,
    "wp-block-button",
    {
      [`has-custom-width wp-block-button__width-${width}`]: width,
      [`has-custom-font-size`]: blockProps.style.fontSize,
    }
  );
  const wrapperClasses = useBlockProps.save({
    className: wrapperClassNames,
  }).className;

  const buttonClasses = classnames(
    className,
    "wp-block-button__link",
    "wp-element-button",
    colorProps.className,
    borderProps.className,
    {
      "no-border-radius": style && style.border && style.border.radius === 0,
    }
  );

  const buttonStyles = {
    ...borderProps.style,
    ...colorProps.style,
    ...spacingProps.style,
  };
  const stylesString = stylesToString(buttonStyles);

  useEffect(() => {
    setAttributes({ buttonStyles: stylesString });
  }, [stylesString]);

  useEffect(() => {
    setAttributes({ buttonClasses });
  }, [buttonClasses]);

  useEffect(() => {
    setAttributes({ wrapperClasses });
  }, [wrapperClasses]);

  function setButtonText(newText) {
    // Remove anchor tags from button text content.
    setAttributes({ text: newText.replace(/<\/?a[^>]*>/g, "") });
  }

  return (
    <div
      {...blockProps}
      className={classnames(wrapperClassNames, blockProps.className)}
    >
      <RichText
        aria-label={__("Button text")}
        placeholder={__("Add text…")}
        value={text}
        onChange={(value) => setButtonText(value)}
        withoutInteractiveFormatting
        className={buttonClasses}
        style={buttonStyles}
        onSplit={(value) =>
          createBlock("itinerator/listing-website-button", {
            ...attributes,
            text: value,
          })
        }
        onReplace={onReplace}
        onMerge={mergeBlocks}
        identifier="text"
      />
    </div>
  );
};

const WidthPanel = () => {
  const {
    attributes: { width: selectedWidth },
    setAttributes,
  } = useEditContext();

  function handleChange(newWidth) {
    const width = selectedWidth === newWidth ? undefined : newWidth;
    setAttributes({ width });
  }

  return (
    <PanelBody title={__("Width settings")}>
      <ButtonGroup aria-label={__("Button width")}>
        {[25, 50, 75, 100].map((widthValue) => {
          return (
            <Button
              key={widthValue}
              isSmall
              variant={widthValue === selectedWidth ? "primary" : undefined}
              onClick={() => handleChange(widthValue)}
            >
              {widthValue}%
            </Button>
          );
        })}
      </ButtonGroup>
    </PanelBody>
  );
};
