import { __ } from "@wordpress/i18n";
import {
  InspectorControls,
  RichText,
  useBlockProps,
} from "@wordpress/block-editor";
import { PanelRow, ToggleControl, SelectControl } from "@wordpress/components";
import { Fragment } from "@wordpress/element";
import {
  Edit,
  LinkedPageSelect,
  ListingCard,
  ListingsDirectory,
  OptionsPanel,
  TagFilters,
  useEditContext,
} from "./common";
import attributes from "../attributes/listings.json";
import { ItineratorBlock } from "./ItineratorBlock";
import { useItinPageMeta, useRecords } from "../hooks";
import { TagCategoryControl } from "./common/TagCategoryControl";
import { useItinSample } from "./store";
import { FilterRecordsRegion } from "./common/FilterRecordsRegion";

new ItineratorBlock("listings", {
  attributes,
  supports: {
    align: true,
    customClassName: true,
  },
  edit: (props) => {
    const { itinPageType, itinPageId } = useItinPageMeta();
    const { sample, loading } = useItinSample(itinPageType, itinPageId);
    if (loading) return null;
    return (
      <Edit {...props}>
        <Sidebar />
        <Display sample={sample} />
      </Edit>
    );
  },
});

const Sidebar = () => {
  const { attributes, setAttributes } = useEditContext();
  const {
    viewType,
    columns,
    showImage,
    showName,
    showCity,
    showDescription,
    filterRecordsByRegion,
  } = attributes;

  return (
    <InspectorControls key="settings">
      <OptionsPanel>
        <LinkedPageSelect type="listing" />
        <PanelRow>
          <ToggleControl
            label="Show Name?"
            checked={showName}
            onChange={(val) => setAttributes({ showName: val })}
          />
        </PanelRow>
        <PanelRow>
          <SelectControl
            label="Style"
            value={viewType}
            onChange={(type) => setAttributes({ viewType: type })}
            options={[
              { value: "alpha", label: "Directory" },
              { value: "badge", label: "Tiles" },
            ]}
          />
        </PanelRow>
        <PanelRow>
          <div style={{ minWidth: "50%" }}>
            <SelectControl
              label="Columns"
              value={columns}
              onChange={(type) => setAttributes({ columns: type })}
              options={[
                { value: "1", label: "1" },
                { value: "2", label: "2" },
                { value: "3", label: "3" },
                { value: "4", label: "4" },
              ]}
            />
          </div>
        </PanelRow>
        {viewType === "alpha" && (
          <Fragment>
            <PanelRow>
              <ToggleControl
                label="Show City?"
                checked={showCity}
                onChange={(val) => setAttributes({ showCity: val })}
              />
            </PanelRow>
          </Fragment>
        )}

        {viewType === "badge" && (
          <Fragment>
            <PanelRow>
              <ToggleControl
                label="Show Image?"
                checked={showImage}
                onChange={(val) => setAttributes({ showImage: val })}
              />
            </PanelRow>
            <PanelRow>
              <ToggleControl
                label="Show Description?"
                checked={showDescription}
                onChange={(val) => setAttributes({ showDescription: val })}
              />
            </PanelRow>
          </Fragment>
        )}
        <FilterRecordsRegion />
        <TagCategoryControl />
      </OptionsPanel>
      <TagFilters />
    </InspectorControls>
  );
};

const Display = ({ sample }) => {
  const { attributes, setAttributes } = useEditContext();
  const { viewType, matchRecordsByTagCategory, buttonText } = attributes;
  const { itinPageId, itinPageType } = useItinPageMeta();
  const id = itinPageId || (sample && sample.id.toString());
  const { records, loading } = useRecords(
    `listings`,
    {
      region_id:
        itinPageType === "region" && !matchRecordsByTagCategory ? id : null,
    },
    id
  );

  const blockProps = useBlockProps({
    className: `itin itinerator-listings itinerator-premium-listings ${attributes.align ? `align${attributes.align}` : ''} ${attributes.customClassName ? `${attributes.customClassName}` : ''}`,
  });


  return (
    <div {...blockProps}>
      <div className="container-fluid">
        {!records.length && !loading && "No Listings found"}
        {viewType === "badge" && (
          <div className="row g-4">
            {records.map((listing) => (
            <ListingCard key={listing.id} listing={listing} />
            ))}
          </div>
        )}
        {viewType === "alpha" && (
          <div className="row">
            <ListingsDirectory records={records} />
          </div>
        )}
      </div>
    </div>
  );
};
