import { useEffect, useRef } from "react";
import { startCase } from "lodash";
import { PluginDocumentSettingPanel } from "@wordpress/edit-post";
import { registerPlugin } from "@wordpress/plugins";
import { useDispatch } from "@wordpress/data";
import { Fragment } from "@wordpress/element";
import {
  Card,
  SelectControl,
  __experimentalVStack as VStack,
  __experimentalText as Text,
} from "@wordpress/components";
import { initialArray } from "../../helpers";
import { useItinPageMeta, usePostType } from "../../hooks";
import { Icon } from "../../assets";
import { useItinOptions } from "../store";
import regionShortcodes from "../../attributes/region-shortcodes.json";
import listingShortcodes from "../../attributes/listing-shortcodes.json";
import eventShortcodes from "../../attributes/event-shortcodes.json";
import challengeShortcodes from "../../attributes/challenge-shortcodes.json";
import itineraryShortcodes from "../../attributes/itinerary-shortcodes.json";
import formShortcodes from "../../attributes/form-shortcodes.json";
// import { PageTagFilters } from "./PageTagFilters";

const SHORTCODES = {
  region: regionShortcodes,
  listing: listingShortcodes,
  event: eventShortcodes,
  challenge: challengeShortcodes,
  itinerary: itineraryShortcodes,
  form: formShortcodes
};

const Panel = () => {
  const type = usePostType();
  const didMountRef = useRef(false);
  const { itinPageType, itinPageId } = useItinPageMeta();
  const { editPost } = useDispatch("core/editor");
  const { options, loading } = useItinOptions(itinPageType);
  useEffect(() => {
    if (!!itinPageType && didMountRef.current) {
      editPost({ meta: { itinerator_page_slug: "" } });
    }
    didMountRef.current = true;
  }, [itinPageType]);
  if (type !== "page") return null;
  const select_options = initialArray("None").concat(
    options.map(({ name, slug, id }) => ({ value: id, label: name }))
  );
  const shortcodes = SHORTCODES[itinPageType];

  return (
    <PluginDocumentSettingPanel
      name="itinerator-sidebar"
      icon={Icon}
      title="Itinerator"
    >
      <SelectControl
        label="Page Type"
        value={itinPageType}
        onChange={(val) =>
          editPost({
            meta: {
              itinerator_page_type: val,
              itinerator_page_id: "",
              itinerator_page_slug: "",
            },
          })
        }
        options={[
          { value: "", label: "None" },
          { value: "region", label: "Single Region" },
          { value: "listing", label: "Single Listing" },
          { value: "itinerary", label: "Single Itinerary" },
          { value: "event", label: "Single Event" },
          { value: "challenge", label: "Single Challenge" },
          { value: "form", label: "Single Form" },
        ]}
      />
      {!!itinPageType && !loading && (
        <SelectControl
          label={`Specific ${startCase(itinPageType)}`}
          value={itinPageId}
          onChange={(val) => {
            const { slug = "" } =
              options.find(({ id }) => !!id && id.toString() === val) || {};
            editPost({
              meta: {
                itinerator_page_id: val,
                itinerator_page_slug: slug,
              },
            });
          }}
          options={select_options}
        />
      )}
      {/* <PageTagFilters /> */}
      {!!shortcodes && (
        <Fragment>
          <Text style={{ display: "block", margin: "0px 0px 8px" }}>
            Available Shortcodes
          </Text>
          <Card style={{ padding: "10px" }}>
            <VStack>
              {shortcodes.map((code) => (
                <code key={code} style={{ fontSize: "13px" }}>{`[itin-${code}]`}</code>
              ))}
            </VStack>
          </Card>
          <Text style={{ display: "block", margin: "8px 0px" }}>
            Available Shortcodes Without Markdown
          </Text>
          <Card style={{ padding: "10px" }}>
            <VStack>
              {shortcodes.map((code) => (
                <code
                  key={code}
                  style={{ fontSize: "13px" }}
                >{`[itin-stripped-${code}]`}</code>
              ))}
            </VStack>
          </Card>
        </Fragment>
      )}
    </PluginDocumentSettingPanel>
  );
};

const { isPlatformAPIKeyValid } = itineratorGlobal;
if (isPlatformAPIKeyValid) {
  registerPlugin("itinerator-sidebar", {
    render: Panel,
    icon: Icon,
  });
}
