import { InspectorControls, useBlockProps, RichText } from "@wordpress/block-editor";
import { useEffect } from "@wordpress/element";
import {
  Edit,
  LinkedPageSelect,
  OptionsPanel,
  TagFilters,
  ToggleControls,
  useEditContext,
} from "./common";
import attributes from "../attributes/regions.json";
import { ItineratorBlock } from "./ItineratorBlock";
import { useItinPageMeta, useRecords } from "../hooks";
import { TagCategoryControl } from "./common/TagCategoryControl";
import { useItinSample } from "./store";

new ItineratorBlock("regions", {
  attributes,
  supports: {
    align: true,
    customClassName: true,
  },
  edit: (props) => {
    const { itinPageType, itinPageId } = useItinPageMeta();
    const { sample, loading } = useItinSample(itinPageType, itinPageId);
    if (loading) return null;
    return (
      <Edit {...props}>
        <Sidebar />
        <Display sample={sample} />
      </Edit>
    );
  },
});

const Sidebar = () => {
  const { attributes } = useEditContext();
  const { itinPageType } = useItinPageMeta();
  const { onlyShowSubregions, ...otherAttributes } = attributes
  return (
    <InspectorControls key="settings">
      <OptionsPanel>
        <ToggleControls custom_attributes={itinPageType === 'region' ? attributes : otherAttributes} />
        <TagCategoryControl />
        <LinkedPageSelect type="region" />
      </OptionsPanel>
      <TagFilters />
    </InspectorControls>
  );
};

const Display = ({ sample }) => {
  const { attributes, setAttributes } = useEditContext();
  const { matchRecordsByTagCategory, onlyShowSubregions, buttonText } = attributes;
  const { itinPageId, itinPageType } = useItinPageMeta();
  const id = itinPageId || (sample && sample.id.toString());
  const { records, loading } = useRecords(
    `regions`,
    {
      parent_region_id:
        itinPageType === "region" && !matchRecordsByTagCategory && onlyShowSubregions ? id : null,
    },
    id
  );

  const blockProps = useBlockProps({
    className: `itin itinerator-regions ${attributes.align ? `align${attributes.align}` : ''} ${attributes.customClassName ? `${attributes.customClassName}` : ''}`,
  });

  const filtered_records = itinPageType === 'region' ? records.filter(r => r.id !== parseInt(id)) : records
  return (
    <div {...blockProps}>
      <div className="container-fluid">
        {!filtered_records.length && !loading && "No Regions found"}
        {filtered_records.map((region) => (
          <div
            key={region.id}
            className={`regions-row row justify-content-end align-items-center searchable-item mb-4 ${region.slug} ${region.parent_slug} ${region.tags.map((t) => `tag-id-${t.tag_id}`).join(" ")}`}
          >
            <div className="col-12 col-md-6 order-md-2 p-0 open-half">
              {region.hero && (
                <div className="image">
                  <img src={region.hero.url} className="img-fluid" />
                </div>
              )}
              <a href={region.path} className="stretched-link"></a>
            </div>
            <div className="col-12 col-md-6 order-md-1 contained-half">
              <div className="content">
                {region.name && <h2>{region.name}</h2>}
                {region.subtitle && attributes.showSubtitle && <h3>{region.subtitle}</h3>}
                {region.short_description  && attributes.showDescription && <p>{region.short_description}</p>}
                {!region.short_description  && attributes.showDescription && region.description && <p>{region.description}</p>}
                {
                  region.name && (
                    <a
                      // href={region.path}
                      className="btn btn-primary"
                      aria-label={`Explore ${region.name}`}
                    ><RichText
                    tagName="span"
                    value={buttonText}
                    onChange={(t) => setAttributes({ buttonText: t })}
                  /> {region.name}
                    </a>
                  )
                }
              </div>
            </div>
          </div>
        ))}
      </div>
    </div>
  );
};
