import { __ } from "@wordpress/i18n";
import { InspectorControls, useBlockProps } from "@wordpress/block-editor";
import { PanelRow } from "@wordpress/components"
import { useItinPageMeta } from "../hooks";
import attributes from "../attributes/related-posts.json";
import { ItineratorBlock } from "./ItineratorBlock";
import { Edit, OptionsPanel, useEditContext } from "./common";
import { useSelect } from "@wordpress/data";
import { useItinSample } from "./store";
import { pluralizeType } from "../helpers";
import { NumberPicker } from "./common/inputs";

new ItineratorBlock("related-posts", {
  attributes,
  edit: (props) => (
    <Edit {...props}>
      <Sidebar />
      <Display />
    </Edit>
  ),
});

const Sidebar = () => {
  return (
    <InspectorControls key="settings">
      <OptionsPanel >
        <PanelRow>
          <NumberPicker label="Max Display Count" attr="displayCount" help="Leave blank to show all." />
        </PanelRow>
      </OptionsPanel>
    </InspectorControls>
  );
};

const Display = () => {
  const { attributes } = useEditContext()
  const { displayCount } = attributes
  const { itinPageType, itinPageId } = useItinPageMeta();
  const blockProps = useBlockProps({
    className: ` itin itinerator-${itinPageType} itinerator-related-posts`,
  });
  const { sample, loading } = useItinSample(itinPageType, itinPageId);
  const id = sample && sample.id;
  const posts = useSelect(
    (select) => {
      if (id) {
        const taxonomy = `itin-${pluralizeType(itinPageType)}`;
        const { getEntityRecords } = select("core");

        const terms = getEntityRecords("taxonomy", taxonomy, {
          per_page: -1,
        });
        const term = !!terms && terms.find((t) => t.slug === id.toString());
        return getEntityRecords("postType", "post", {
          per_page: typeof displayCount === 'number' ? displayCount : -1,
          order_by: "date",
          _embed: true,
          [taxonomy]: [term.id],
        });
      }
    },
    [id, itinPageType, displayCount]
  );
  if (loading) return null;
  return (
    <div {...blockProps}>
      {posts && !!posts.length ? (
        posts.map((post) => {
          return <RelatedPost post={post} />;
        })
      ) : (
        <div>
          {!!itinPageType
            ? "No related posts found"
            : "This block can only be used with an Itinerator page type set."}
        </div>
      )}
    </div>
  );
};

const RelatedPost = ({ post: { _embedded, title, excerpt } }) => {
  const media = _embedded["wp:featuredmedia"];
  const media_sizes = media &&
    !!media.length &&
    media[0].media_details.sizes
  const size = Object.keys(media_sizes).includes('medium_large') ? 'medium_large' : 'full'
  const url = media_sizes[size].source_url;

  return (
    <div className="itinerator-related-post">
      <img
        src={url}
        style={{ height: 130, width: "100%", objectFit: "cover" }}
      />
      <h2>{title.rendered}</h2>
      <div
        dangerouslySetInnerHTML={{
          __html: excerpt.rendered,
        }}
      />
    </div>
  );
};
