import { createReduxStore, register, resolveSelect } from "@wordpress/data";
import { useState, useEffect } from "@wordpress/element";
import { reduce } from "lodash";
import { getWpApiPath, pluralizeType } from "../../helpers";
const actions = {
  setSample(sample, sample_type) {
    return {
      type: "SET_SAMPLE",
      sample_type,
      sample,
    };
  },
  setLocationData(data, key) {
    return {
      type: "SET_LOCATION_DATA",
      data,
      key,
    };
  },
  setOptions(options, page_type) {
    return {
      type: "SET_OPTIONS",
      options,
      page_type,
    };
  },
  fetchFromAPI(path, params) {
    return {
      type: "FETCH_FROM_API",
      path,
      params,
    };
  },
};

export const itineratorStore = createReduxStore("itinerator-sample-data", {
  reducer(state = { samples: {}, options: {}, location_data: {} }, action) {
    switch (action.type) {
      case "SET_SAMPLE":
        return {
          ...state,
          samples: {
            ...state.samples,
            [action.sample_type]: action.sample,
          },
        };
      case "SET_LOCATION_DATA":
        return {
          ...state,
          location_data: {
            ...state.location_data,
            [action.key]: action.data,
          },
        };
      case "SET_OPTIONS":
        return {
          ...state,
          options: {
            ...state.options,
            [action.page_type]: action.options,
          },
        };
      default:
        return state;
    }
  },
  actions,
  selectors: {
    getSample(state, sample_type) {
      return state.samples[sample_type];
    },
    getLocationData(state, page_type, params, id) {
      const key = [page_type, JSON.stringify(params), id].join("-");
      return state.location_data[key];
    },
    getOptions(state, page_type) {
      return state.options[page_type];
    },
  },
  controls: {
    FETCH_FROM_API(action) {
      const filtered_params =
        !!action.params &&
        reduce(
          action.params,
          (object, value, key) => {
            if (typeof value === "boolean" || (!!value && !!value.length))
              object[key] = value;
            return object;
          },
          {}
        );
      const paramString = new URLSearchParams(filtered_params).toString();
      return fetch(getWpApiPath(action.path) + "?" + paramString).then(
        (res) => {
          if (res.status === 404) return null;
          return res.json();
        }
      );
    },
  },
  resolvers: {
    *getSample(sample_type, id) {
      const sample = yield !!sample_type &&
        actions.fetchFromAPI(`${pluralizeType(sample_type)}/${id || "sample"}`);
      return actions.setSample(sample, sample_type);
    },
    *getLocationData(page_type, params, id) {
      const key = [page_type, JSON.stringify(params), id].join("-");
      const data = yield actions.fetchFromAPI(
        !!page_type
          ? `${pluralizeType(page_type)}/${id || "sample"}/location`
          : "locations",
        params
      );
      return actions.setLocationData(data, key);
    },
    *getOptions(page_type) {
      const options = yield actions.fetchFromAPI(
        `record_list_by_type/${pluralizeType(page_type)}`
      );
      return actions.setOptions(options, page_type);
    },
  },
});

register(itineratorStore);

export const useItinSample = (type, id) => {
  const [sample, setSample] = useState();
  const [loading, setLoading] = useState(true);
  useEffect(() => {
    resolveSelect(itineratorStore)
      .getSample(type, id)
      .then(setSample)
      .then(() => setLoading(false));
  }, [type, id])
  return { sample, loading };
};

export const useItinLocationData = (page_type, params, id) => {
  const [locationData, setLocationData] = useState();
  const [loading, setLoading] = useState(true);
  useEffect(() => {
    resolveSelect(itineratorStore)
    .getLocationData(page_type, params, id)
    .then(setLocationData)
    .then(() => setLoading(false));
  }, [page_type, params, id])
  return { locationData, loading };
};

export const useItinOptions = (page_type) => {
  const [options, setOptions] = useState([]);
  const [loading, setLoading] = useState(true);
  useEffect(() => {
    if (!!page_type) {
      resolveSelect(itineratorStore)
      .getOptions(page_type)
        .then(setOptions)
      .then(() => setLoading(false));
    }
  }, [page_type])
  return { options, loading };
};
