import { useEffect, useState, useRef } from "react";
import { useSelect } from "@wordpress/data";
import apiFetch from "@wordpress/api-fetch";
import {
  getPosition,
  getWpApiPath,
  initialArray,
  listToSelectOptions,
} from "./helpers";
import { reduce, startCase } from "lodash";
import { useEditContext } from "./editor_blocks/common";
import { useItinLocationData } from "./editor_blocks/store";

/**
 * Custom hook to retrieve the current post type from the block editor.
 *
 * @return {string} The current post type.
 */
export const usePostType = () => {
  // Use the 'useSelect' hook to access the current post type from the block editor
  const type = useSelect(
    (select) => select("core/editor").getEditedPostAttribute("type"),
    []
  );
  return type;
};

/**
 * Custom hook to get meta information about the current page.
 *
 * @return {object} An object containing page meta information.
 */
export const useItinPageMeta = () => {
  // Use the 'useSelect' hook to access meta information from the block editor
  const { itinerator_page_type, itinerator_page_slug, itinerator_page_id } =
    useSelect(
      (select) => select("core/editor").getEditedPostAttribute("meta"),
      []
    ) || {};
  return {
    itinPageType: itinerator_page_type,
    itinPageSlug: itinerator_page_slug,
    itinPageId: itinerator_page_id,
  };
};

/**
 * Custom hook to fetch and store information about itinerary pages.
 *
 * @param {string} type - The type of itinerary page to retrieve.
 * @return {array} An array of itinerary pages.
 */
export const useItinPages = (type) => {
  const [pages, setPages] = useState([]);
  useEffect(() => {
    apiFetch({ path: `itinerator/v1/itin_pages/${type || ""}` })
      .then((data) => {
        setPages(data);
      })
      .catch((data) => {
        console.log("useItinPages Error:", data);
      });
  }, [type]);
  return pages;
};

/**
 * Custom hook to work with attributes in the block editor.
 *
 * @param {string} attr - The name of the attribute to work with.
 * @return {array} An array containing the attribute, setter, and name.
 */
export const useAttribute = (attr) => {
  const { attributes, setAttributes } = useEditContext();
  const name = startCase(attr);
  const attribute = attributes[attr];
  const setAttribute = (val) => {
    setAttributes({ [attr]: val });
  };
  return [attribute, setAttribute, name];
};

/**
 * Custom hook to get the previous value of a variable.
 *
 * @param {any} value - The variable for which to retrieve the previous value.
 * @return {any} The previous value of the variable.
 */
export const usePrevious = (value) => {
  // Create a reference to store the previous value
  const ref = useRef();

  // Use the 'useEffect' hook to update the reference with the current value
  useEffect(() => {
    ref.current = value;
  }, [value]);

  // Return the previous value from the reference
  return ref.current;
};

/**
 * Custom hook to fetch and store tag categories.
 *
 * @return {object} An object containing default category and options.
 */
export const useTagCategories = () => {
  // Initialize state variables for the default category and options
  const [defaultCategory, setDefaultCategory] = useState("");
  const [options, setOptions] = useState([]);


  // Use the 'useEffect' hook to make an API request when the component mounts
  useEffect(() => {
    // Fetch tag categories from the specified API endpoint
    fetch(getWpApiPath("tag_categories"))
      .then((res) => res.json())
      .then((categories) => {
        if (categories.length > 0) {
          // Set the default category to the first category
          setDefaultCategory(categories[0]);
        } else {
          // Handle the case where there are no categories
          setDefaultCategory("");
        }

        // Set the options by converting the categories into select options
        setOptions(listToSelectOptions(categories));
      })
      .catch((error) => {
        console.error('Error fetching tag category:', error);
        // Handle the error, e.g., display an error message to the user.
      });
  }, []);

  // Return an object containing the default category and options
  return { defaultCategory, options };
};


/**
 * Custom hook to fetch and store tag options by category.
 *
 * @param {string} category - The category for which to fetch tag options.
 * @param {string} placeholder - The placeholder value for tag options.
 * @return {object} An object containing tag options and the default option.
 */
export const useTagsByCategoryOptions = (category, placeholder) => {
  // Initialize state variables for the default option and options
  const [defaultOption, setDefaultOption] = useState("");
  const [options, setOptions] = useState([]);

  // Concatenate the initial array with the fetched options to form tagOptions
  const tagOptions = initialArray(placeholder).concat(options);

  // Use the 'useEffect' hook to make an API request when the category changes
  useEffect(() => {
    if (!!category) {
      // Fetch tag options based on the provided category
      fetch(getWpApiPath(`tags_by_category/${category}`))
        .then((res) => res.json())
        .then((tags) => {
          if (Array.isArray(tags) && tags.length > 0) {
            // Set the default option to the first tag's ID as a string
            setDefaultOption(tags[0].id.toString());

            // Set the options by mapping tag data to value-label pairs
            setOptions(
             tags.map(({ name, id }) => ({ value: id.toString(), label: name }))
            );
          } else {
            // Handle the case where tags is not an array or is empty
            setDefaultOption("1");
            setOptions([]);
          }
        })
        .catch((error) => {
          console.error('Error fetching tags:', error);
          // Handle the error, e.g., display an error message to the user.
        });
    }
  }, [category]);

  // Return an object containing tag options and the default option
  return { tagOptions, defaultOption };
};


/**
 * Custom hook to fetch and store record list options.
 *
 * @param {string} type - The type of record list for which to fetch options.
 * @param {string} placeholder - The placeholder value for record list options.
 * @return {array} An array of record list options.
 */
export const useRecordListOptions = (type, placeholder) => {
  // Initialize state variables for options and an initial value
  const [options, setOptions] = useState([]);
  const init = initialArray(placeholder);

  // Use the 'useEffect' hook to make an API request when the type changes
  useEffect(() => {
    // Fetch record list options based on the provided type
    fetch(getWpApiPath(`record_list_by_type/${type}`))
      .then((res) => res.json())
      .then((list) =>
        setOptions(list.map(({ name, slug }) => ({ value: slug, label: name }))
        ));
  }, [type]);

  // Return an array that concatenates the initial value and fetched options
  return init.concat(options);
};


/**
 * Custom hook to fetch and store records.
 *
 * @param {string} type - The type of records to fetch.
 * @param {object} params - Additional parameters for fetching records.
 * @param {string} id - The ID for matching records.
 * @return {object} An object containing records and loading status.
 */
export const useRecords = (type, params, id) => {
  // Access the attributes from the block editor context
  const { attributes } = useEditContext();

  // Retrieve tagCategory and matchRecordsByTagCategory from attributes
  const { tagCategory, matchRecordsByTagCategory } = attributes;

  // Get tagIds using another custom hook
  const tagIds = useTagIds();

  // Initialize state variables for records and loading status
  const [records, setRecords] = useState([]);
  const [loading, setLoading] = useState(true);

  // Define default parameters for the API request
  const default_params = {
    tagIds,
    tagName: matchRecordsByTagCategory ? id : null,
    tagCategory: matchRecordsByTagCategory ? tagCategory : null,
  };

  // Combine default parameters with any additional parameters passed
  const filtered_params = reduce(
    { ...default_params, ...params },
    (object, value, key) => {
      if (!!value && !!value.length) object[key] = value;
      return object;
    },
    {}
  );

  // Create a query parameter string from the filtered parameters
  const paramString = new URLSearchParams(filtered_params).toString();
  // Use the 'useEffect' hook to make the API request when type or parameters change
  useEffect(() => {
    fetch(getWpApiPath(type) + "?" + paramString)
      .then(res => {
        if (!res.ok) {
          console.error("Request failed with status " + res.status);
          throw new Error("HTTP status " + res.status);
        }
        return res.json();
      })
      .then(setRecords)
      .catch(err => console.error("Fetch Error: ", err))
      .finally(() => setLoading(false));  // use finally to ensure loading is set to false even on error
  }, [type, paramString]);
  // Return an object containing records and loading status
  return { records, loading };
};


/**
 * Custom hook to fetch and store additional listings.
 *
 * This custom hook is designed to retrieve additional listings from an API using the `fetch` function.
 * It returns an object with two properties: 'records' (an array of listings) and 'loading' (a boolean
 * flag to indicate whether the data is still being loaded).
 *
 * @return {object} An object containing additional listings and loading status.
 */
export const useAdditionalListings = () => {
  // Initialize 'records' state as an empty array and 'loading' state as 'true'.
  const [records, setRecords] = useState([]);
  const [loading, setLoading] = useState(true);

  // Use the 'useEffect' hook to fetch data from the API when the component mounts.
  useEffect(() => {
    // Fetch additional listings from the API and parse the response as JSON.
    fetch(getWpApiPath("additional_listings"))
      .then((res) => res.json())
      .then(setRecords) // Set the 'records' state with the fetched data.
      .then(() => setLoading(false)); // Set 'loading' to 'false' once data is loaded.
  }, []); // The empty dependency array ensures this effect runs only once.

  // Return an object containing 'records' and 'loading'.
  return { records, loading };
};


/**
 * Custom hook to perform an effect after the component has mounted.
 *
 * This custom hook is designed to execute a specified callback function after the component
 * has mounted, which can be useful for performing side effects or initializations that should
 * occur only once. It uses the 'useEffect' hook with a dependency array to ensure the callback
 * runs after the component is mounted.
 *
 * @param {function} callback - The function to execute after mounting.
 * @param {array} deps - Dependency array for the effect.
 */
export const useAfterMountEffect = (callback, deps) => {
  // Create a 'didMountRef' using 'useRef' to track whether the component has already mounted.
  const didMountRef = useRef(false);

  // Use the 'useEffect' hook with the specified 'deps' array.
  useEffect(() => {
    // If the component has already mounted, execute the 'callback'.
    if (didMountRef.current) {
      callback();
    } else {
      // If it's the initial mount, set 'didMountRef' to 'true' to mark that the component
      // has now mounted.
      didMountRef.current = true;
    }
  }, deps);
};


/**
 * Custom hook to get tag IDs from attributes.
 *
 * This custom hook is designed to extract tag IDs from the 'attributes' object using the
 * 'useEditContext' hook. It specifically retrieves the 'tagFilters' property from the
 * 'attributes' and returns an array of tag IDs, or 'null' if there are no tag filters available.
 *
 * @return {array|null} An array of tag IDs or null if no tags are available.
 */
export const useTagIds = () => {
  // Retrieve the 'attributes' object using the 'useEditContext' hook.
  const { attributes } = useEditContext();

  // Extract the 'tagFilters' property from the 'attributes'.
  const { tagFilters } = attributes;

  // If 'tagFilters' is not defined or is an empty array, return [].
  if (!tagFilters || !tagFilters.length) return [];

  // Map the 'tagFilters' to extract and return an array of tag IDs.
  return tagFilters.map((t) => t.id);
};


/**
 * Custom hook to work with location data.
 *
 * This custom hook is designed to handle location data within a specific context. It extracts
 * relevant data from the 'attributes' object using the 'useEditContext' hook, and it also leverages
 * the 'useTagIds' hook. The primary purpose is to retrieve and structure location data, providing an
 * object containing location information and loading status.
 *
 * @return {object} An object containing locations and loading status.
 */
export const useLocation = () => {
  // Extract attributes from the 'useEditContext' hook.
  const { attributes } = useEditContext();
  const { pageType, recordId } = attributes;
  const tagIds = useTagIds();

  // Define functions to retrieve show parameters.
  const getShowsByName = (name) => ({
    showMarker: attributes[`show${name}Markers`],
    showBoundaries: attributes[`show${name}Boundaries`],
    showLabel: attributes[`show${name}Labels`],
  });

  const getShowParams = () => {
    const res = {};
    ["Regions", "Listings"].forEach((name) => {
      const { showMarker, showBoundaries, showLabel } = getShowsByName(name);
      res[`show${name}`] = showMarker || showBoundaries || showLabel;
    });
    return res;
  };

  // Retrieve location data and loading status using 'useItinLocationData'.
  const { locationData = {}, loading } = useItinLocationData(
    pageType,
    { tagIds, ...getShowParams() },
    recordId || ""
  );

  // Initialize an array to store location data.
  const locations = [];

  // Function to extract attributes from location data.
  const getAttributes = (type, location) => {
    const type_a = type.split("_");
    const type_name = type_a[0];
    const name = startCase(type_name);
    const plural = type_a[1][8] || "";
    const attribute_name = name + plural;
    return {
      key: location.id,
      location,
      position: getPosition(location), // The 'getPosition' function is not defined in this code.
      ...getShowsByName(attribute_name),
      type: type_name,
    };
  };

  // Populate the 'locations' array with extracted location attributes.
  Object.keys(locationData).forEach((key) => {
    const value = locationData[key];
    if (Array.isArray(value)) {
      value.forEach((location) => {
        locations.push(getAttributes(key, location));
      });
    } else {
      locations.push(getAttributes(key, value));
    }
  });

  // Return an object containing 'locations' and 'loading' status.
  return { locations, loading };
};


/**
 * Custom hook to fetch and store linked page information.
 *
 * This custom hook is designed to fetch information about a linked page using the provided 'key'
 * to access the linked page ID from the 'attributes' object obtained through the 'useEditContext' hook.
 * It retrieves data about the linked page by making a request to a WordPress REST API endpoint and
 * stores this information in the 'page' state. The 'key' parameter allows for flexibility in accessing
 * different linked pages within the component.
 *
 * @param {string} key - The key used to access the linked page ID in attributes.
 * @return {object} An object containing information about the linked page, or 'undefined' if no page is linked.
 */
export const useLinkedPage = (key = "linkedPageId") => {
  // Extract 'attributes' from the 'useEditContext' hook.
  const { attributes } = useEditContext();

  // Obtain the linked page ID using the provided 'key'.
  const linkedPageId = attributes[key];

  // Initialize 'page' state.
  const [page, setPage] = useState();

  // Use the 'useEffect' hook to fetch page information when 'linkedPageId' changes.
  useEffect(() => {
    if (linkedPageId) {
      // Fetch data about the linked page from the WordPress REST API.
      fetch(`/wp-json/wp/v2/pages/${linkedPageId}`)
        .then((res) => res.json())
        .then(setPage);
    }
  }, [linkedPageId]);

  // Return the 'page' object, which contains information about the linked page, or 'undefined' if no page is linked.
  return page;
};

